/*!
  \file overview.hpp This file contains documentation only, and is not part of the MynahSA code base.
*/

/*!

\page overview Overview
\section over1 Introduction
    <strong>MynahSA</strong> is used to store and restore data for C++ object on streams.  The streams support file I/O and 
       network communications.

   Conceptually, <strong>MynahSA</strong> is broken down into three levels of abstraction:
   
   \image html concept_diag.png "Conceptual Diagram "width=211
   \image latex concept_diag.png "Conceptual Diagram" width=5cm

\section over2 Stream level

   The streaming classes act as an interface between <strong>MynahSA</strong>'s archiving system and the physical data stream.  There are five different types of stream: <strong>TextStream</strong>, <strong>BinaryStream</strong>, <strong>SSLStream</strong>, <strong>TCPStream</strong> and <strong>BlowfishStream</strong>.  These classes provide a structured input/output mechanism for the C/C++ built-in types (e.g. char, float, int ...) and <strong>std::string</strong>.

   Stream classes are uni-directional, and the direction is identified by either an "I" for input or an "O" for  output class name prefix.  For example, <strong>MynahSA::IBlowfishStream</strong> denotes an input stream using the Blowfish algorithm for decryption.

   MyanhSA's stream classes differ from the standard library streams in that they are designed for recovering data exactly as it was stored.  For example, if a <strong>std::string</strong> containing spaces is written to a standard library stream the corresponding stream read (in-stream) operation will stop reading when it encounters a space or line-break.  This behavior is expected because the standard library streams are designed for generating user readable text.  In contrast, <strong>MynahSA</strong>'s stream operators are not designed  for readability.  Control information is stored on the stream allowing the in-stream operator to recover the exact string that was stored.

   The choice of stream type will vary depending on application - for file storage, Text, Binary or Blowfish streams may be used.  For communications over the network, SSL and TCP streams are used.  Naturally, the SSL and Blowfish streams provide data encryption - the other streams do not.

\section over3 Archive level
   At the archive level, one class, <strong>MynahSA::Archive</strong> serves as the main interface between user classes and <strong>MynahSA</strong>'s stream classes.

   There is little design choice at the archive level:  <strong>MynahSA::Archive</strong> has two descendants: <strong>IArchive</strong> for input and <strong>OArchive</strong> for output, which must be bound respectively onto an input or output stream instance. 

   Class <strong>MynahSA::Archive</strong> uses template meta-programming to classify types as either simple types (C/C++ built-in types), enumerations (<strong>enum</strong>), STL containers or user defined types.  Types are classified at compile time and one of the following actions are chosen to handle each object: 
   <ul>
      <li>C++ Built-in types and <strong>std::string</strong>s are passed directly between the stream  and user 
          class layers.</li>
      <li>Enumerations in the User Class layer converted to integers at the stream layer.</li>
      <li>STL Containers are recursively broken down into simple types and passed between the layers.</li>
      <li>User defined types are archived by calling the user provided <strong>serialize</strong> method.</li>
   </ul>      

\section over4 User Classes

    User classes interact with <strong>MynahSA::Archive</strong> by providing a method named <strong>serialize</strong>.  The serialize method is symmetric - meaning it is responsible for both storing and restoring data from streams.  Consider the following object definition:

\code
class MyObject { 
public:
  MyObject();
  ~MyObject();
  void serialize(MynahSA::Archive& ar) {
      ar & _someInteger;
      ar & _someString;
  };
private:
  int _someInteger;
  std::string _someString;
};
\endcode
   The <strong>serialize</strong> method invokes <strong>MynahSA::Archive::operator&</strong> on each of the class data members.  The <strong>operator&</strong> is overloaded allowing for the storage of all built-in C/C++ types, several common Standard Template Library (STL) containers, and C++ structs and classes that provide a <strong>serialize</strong> method.

\section over5 Pointer objects

   <strong>MynahSA</strong> supports shared pointer objects.  C/C++ pointer types are not supported as a matter of design decision.  <strong>MynahSA</strong> provides a basic shared pointer implementation named <strong>SharedPtr</strong>.  The <strong>MynahSA</strong> library may be specialized at compile time to use a project's specific shared pointer type.

   The reason for forcing shared pointer usage is to prevent incomplete pointer life-cycle planning.  All to often C++ projects fall foul of bad pointer programming practice leaving memory leaks, double deletion problems and segmentation faults to <a href="http://www.valgrind.org">Valgrind</a> and frustrated programmers to find.

   With shared pointers most of the heap allocated object life cycle can be safely ignored - just throw the pointer away when it is no longer needed.

\subsection spparticulars Shared Pointer Particulars

   For <strong>MynahSA</strong> to store an object by shared pointer the object must derive from base class <strong>MynahSA::IoBase</strong>.  Class <strong>MynahSA::IoBase</strong> acts as a pure base class requiring two members: <strong>serialize</strong>, which connects user defined types with <strong>MynahSA</strong> and <strong>ioName</strong>, which returns the name of the object on the stream.  The <strong>ioName</strong> method is used by <strong>MynahSA::Archive</strong> to identify a specific object instance from a potential class hierarchy.

   Restoring objects by a shared pointer requires that a constructor object be registered with the <strong>MynahSA::IArchive</strong> instance <b>prior</b> to attempting to restore the object.  The constructor object allows the input archive to invoke a "virtual constructor" for the specific object type, instantiate the object type and invoke the correct <strong>serialize</strong> method associated with the object instance.  This process ensures that storing and restoring of objects results in the correct object hierarchy.

   When a shared pointer to an object is stored the entire object contents are written onto the stream.  If the same pointer is stored a second time a Unique Pointer Reference (UPR) is stored in its place.  UPRs allow graph structures to be stored without ending up in infinite loops.  Graphs with circular links are converted into trees by replacing back-edges with a UPR.

\section over6 Theory of operation

   Unlike XML, the data stored on streams using <strong>MynahSA</strong> does not contain explicit formatting information.  There are no tags to identify the beginning or end of data members and there are no identifying names associated with data members.  The data streams produced by <strong>MynahSA</strong> have no context on their own - the context only exists in the presence of the source code used to archive the respective objects.  This approach has two key benefits:

   <ul>
     <li>Objects require significantly less storage on disk or network bandwidth.</li>
     <li>Encoding and decoding requires significantly less CPU time and memory bandwidth.</li>
   </ul>

   The disadvantage is that the archives are not human readable, nor easily human modifiable.  Furthermore, the streams are fragile - a byte inserted incorrectly will cause the stream to become corrupt and invalidate the data contained in the remainder of the stream.  For most applications these are not serious concerns, however, care must be taken to ensure that the data streams are robustly stored.

\section over7 Other toolkits and History

   Several other toolkits are available that warrant note and examination:
   <ul>
     <li><a href="http://s11n.net">s11n</a> - a toolkit for serializing C++ objects.  Provides XML storage.</li>
     <li><a href="http://www.boost.org/libs/serialization/doc/index.html">Boost::Serialization</a> - Boost's serialization toolkit.</li>
   </ul>

   <strong>MynahSA</strong> was developed as an alternative to Boost's <a href="http://www.boost.org/libs/serialization/doc/index.html">serialization</a> toolkit.  In fact, Boost::Serialization was in use for two internal projects at Mynah-Software prior to the decision to engineer <strong>MynahSA</strong>.  Both projects communicated via SSL and used Boost::Serialization to manage object serialization.  <strong>MynahSA</strong> was developed to work around three technical difficulties:

   <ul>
      <li>Inherent buffering of data: To send data, objects were first serialized onto a buffer, then the buffer contents were transferred over the network.  The receiving end performed the opposite: read data from the network into a buffer, then restore the objects.  Excessive copying was eliminated in the implementation of <strong>MynahSA::ISSLStream</strong> and <strong>MynahSA::OSSLStream</strong>.</li>
      <li>Confusing object constructor behavior: To serialize objects by shared pointer Boost::Serialization uses a cool mechanism whereby constructors are registered through global object construction.  In practice, GCC places the constructor registration code in the ELF ini/fini sections, which introduces a practical problem: ini/fini sections are executed in order determined by link order!  For most programs this approach works fine, however, Mynah-Software (and its customers) were repeatedly stumped by problems related to link ordering. <strong>MynahSA</strong> uses explicit object registration to avoid this problem.</li>
      <li>Template complexity and ease of learning: <strong>MynahSA</strong> was engineered for a single purpose - producing client/server applications in C++ with a minimal amount of effort.  Learning, maintaining and enhancing code with complex template mechanics eats programmer time.  <strong>MynahSA</strong> has been engineered to avoid this.</li>
  </ul>

*/
