/*!
  \file tutorial2.hpp This file contains documentation only, and is not part of the MynahSA code base.
*/

/*!

\page tutorial2 Tutorial 2: Symmetric Encryption For File I/O
\section tut2overview Overview
      MynahSA provides input and output streams that are encrypted using OpenSSL's implementation of <a href="http://www.counterpane.com/blowfish.html">Counterpane's Blowfish</a> cipher.  Using the Mynahsa::IBlowfishStream and MynahSA::OBlowfishStream classes data can be encrypted on disk files or std::streams.

      Strong encryption is particularly helpful for applications where your data needs to be secure - this mechanism enforces security because data cannot be read or written without encryption.

      Unlike simple password protections that prevent a file from being loaded, files stored with the Blowfish stream are completely encrypted.  Barring an attack on Blowfish or knowing the password it is extremely unlikely that the data will be extracted from an encrypted file.

\section tut2usage Usage

      The Blowfish streams are used similarly to the standard input/output streams presented in \ref tutorial1, however, a password must be supplied to the class constructor.<br><br>
	To demonstrate the use of the Blowfish streams, a simple C++ struct will be archived.  Three steps will be performed: 
      <ul>
	<li>Step 1: Create an OBlowfishStream and archive the data.</li>
	<li>Step 2: Display the contents of the archived data.</li>
	<li>Step 3: Create an IBlowfishStream and restore data structure.</li>
      </ul>
      The data structure definition is the following:
\code
struct SimpleContainer { 
  void serialize(Archive&amp ar) { 
    ar &amp _name;
    ar &amp _age;
  }

  string _name;
  int _age;
};
\endcode
      The first step is to create an instance of the SimpleContainer and fill it with some data, then store it onto an output Blowfish stream using the password "Private Key".
      \code
ostringstream ostr;
{ 
  SimpleContainer sc;
  sc._name = "Brett";
  sc._age = 32;
    
  OBlowfishStream obfs(ostr, "Private Key");
  OArchive&lt;OBlowfishStream&gt oabfs(obfs);
   
  oabfs << sc;
}
\endcode
      To examine the data on the stream each character is displayed it as an integer; preventing control characters from affecting the console.
      \code
for (unsigned int i=0; i&lt;ostr.str().size(); i++) { 
  unsigned char ch = ostr.str()[i];
  cout << "Stream[" << i << "] contains value: " << ((unsigned int) ch) 
       << endl;
}
cout << endl;
\endcode
      This produces the output:
\verbatim
Stream[0] contains value: 203
Stream[1] contains value: 41
Stream[2] contains value: 151
Stream[3] contains value: 189
Stream[4] contains value: 126
Stream[5] contains value: 126
Stream[6] contains value: 5
Stream[7] contains value: 196
Stream[8] contains value: 1
Stream[9] contains value: 25
Stream[10] contains value: 102
Stream[11] contains value: 51
Stream[12] contains value: 198
Stream[13] contains value: 26
Stream[14] contains value: 149
Stream[15] contains value: 212
\endverbatim
      These numbers are garbled and exactly what is expected from an encryption routine.<br><br>
	To decrypt and restore the object state the exact opposite process is performed:
\code
{
  SimpleContainer sc;
  istringstream istr(ostr.str());
  IBlowfishStream ibfs(istr, "Private Key");
  IArchive&lt;IBlowfishStream&gt iabfs(ibfs);
  iabfs >> sc;
  cout << "sc's name: " << sc._name << endl;
  cout << "sc's age: " << sc._age << endl; 
}
\endcode
      Which produces the expected results:
\verbatim
sc's name: Brett
sc's age: 32
\endverbatim
\section tut2notes Notes
      <ul>
	<li>The current implementation of IBlowfishStream does no checking of the validity of passwords.  Because Blowfish will happily decrypt gibberish, attempting to recover a stream with an incorrect password is considered an error, and the results are unlikely to be correct.  A future version of MynahSA will address this issue.</li>
	<li>Stream closing: OBlowfishStream is a buffering stream that collects chunks of 8 bytes at a time.  After eight bytes are accumulated they are encrypted and written onto the output stream.  If your data is less then 8 bytes, be sure to close the stream prior to examining the data on the stream.  OBlowfishStream's destructor automatically closes the stream.</li>
	<li>Key strength is a function of its length.  Please see <a href="http://www.openssl.org/docs/crypto/blowfish.html#">OpenSSL's Blowfish notes</a> for details.</li>
      </ul>
*/
