/*!
  \file tutorial3.hpp This file contains documentation only, and is not part of the MynahSA code base.
*/


/*!

\page tutorial3 Tutorial 3: TCP Client/Server Demonstration

\section tut3overview Overview

      Tutorial 3 demonstrates the creation of a simple multi-threaded client/server application using MynahSA's TCP transport mechanism.  The process of setting up a multi-threaded server is straight forward, and builds on the basics established in \ref tutorial1.<br><br>
In this example, the client will transmit a "Ping" message to a server and the server will respond back to the client with a ping response.  The behavior is analogous to the unix ping command and measurements of the round trip time are reported.  The protocol diagram is:

   \image html ping_proto.png "Ping Protocol"
   \image latex ping_proto.png "Ping Protocol" width=5cm

\section tut3netobjimp Network Object Implementation

      The server in this application has no state, which simplifies the implementation. <br><br>
	To start with, the objects transmitted across the network are defined.  The <strong>RequestPing</strong> object contains a time stamp, and its definition (from examples/tutorial3/requestping.hpp) is as follows:

\code
class RequestPing : public MynahSA::IoBase { 
public:
  void serialize(MynahSA::Archive& ar) { 
    ar & _sec;   // serialize seconds
    ar & _usec;  // serialize useconds
  }
  
  //! default constructor - required to instantiate!
  RequestPing();

  //! construction for restoration off of stream
  RequestPing(const struct timeval& tv);

  //! destructor
  virtual ~RequestPing();

  //! identify name of request object
  virtual std::string ioName() const;
  
    //! allow user read access to seconds
  inline unsigned int getSeconds() const { return _sec; }
  
  //! allow user read access to microseconds
  inline unsigned int getUSeconds() const { return _usec; }
  
private:
  unsigned int _sec;
  unsigned int _usec;
};
\endcode

      The <strong>ResponePing</strong> object that the server will transmit back to the client is similar, however, a boolean status flag named <strong>_ok</strong> is added to indicate the server's status.  The definition for <strong>ResposnePing</strong> is placed in examples/tutorial3/responseping.hpp.
\code
class ResponsePing : public MynahSA::IoBase { 
public:

  void serialize(MynahSA::Archive& ar) { 
    ar & _ok;
    ar & _sec;
    ar & _usec;
  }

  //! construction for restoration off of stream
  ResponsePing(bool ok=false, unsigned int sec=0, unsigned int usec=0);
  
  //! destructor
  virtual ~ResponsePing();

  //! identify name of the response object
  virtual std::string ioName() const;

  //! return true - indicates server is ok.  False; otherwise
  bool isOk() const { return _ok; }

  //! allow user read access to seconds
  inline unsigned int getSeconds() const { return _sec; }
  
  //! allow user read access to microseconds
  inline unsigned int getUSeconds() const { return _usec; }

private:
  //! indicate to the client that everything is ok here.
  bool _ok;
  
  //! seconds recoreded at client
  int _sec;
  
  //! microseconds recorded at client
  int _usec;
};
\endcode
      Both of these classes are simple container classes and serve only as a mechanism for <strong>MynahSA</strong>'s archiving system.
      \section Network Server Implementation
      The next step is to define the "Server" behavior of the system.  The server is nothing more then a mechanism that converts one descendant class of MynahSA::IoBase into another - in this case, reading a RequestPing object and returning a ResponsePing.  The server behavior is defined by creating a class that derives from MynahSA::RequestResponseHandler.  The following code segment is taken from MynahSA::RequestResponseHandler, and shows the <strong>operator()</strong> method that must be overridden to handle the <strong>RequestPing</strong> object.
\code
class RequestResponseHandler { 
  public:
    ...
    //! Request -> Response handler
    virtual SHARED_PTR<IoBase> 
      operator()(const SHARED_PTR<IoBase>&) = 0;  
};
\endcode

      MynahSA's network server classes invoke RequestResponseHandler::operator() every time an object is received.  To handle ping objects, an object named <strong>PingRequestResponseHandler</strong> is derived from MynahSA::RequestResponseHandler, and its <strong>operator()</strong> is:

\code
SHARED_PTR<IoBase> 
PingRequestResponseHandler::operator()(const SHARED_PTR<IoBase>& req) { 
  SHARED_PTR<IoBase> resp;

  if (req->ioName() == RequestPing().ioName()) { 
    SHARED_PTR<RequestPing> prq = static_pointer_cast<RequestPing>(req);
    // create the response and copy the seconds and microseconds parameters 
    //  from the request object
    resp = SHARED_PTR<IoBase>(new ResponsePing(true,
                                               prq->getSeconds(),
                                               prq->getUSeconds()));
  } else {
    // this is a logic error; unknown class inserted! 
    cerr << "PingRequestResponseHandler::operator()"
            " : Unknown class received." << endl;
  }
    
  return resp;
}
\endcode
      There are a few things of note here: 
      <ul>
	<li>This server is completely stateless.  There is no sharing of information between two different connections, and because of this, the implementation of <strong>PingRequestResponseHandler</strong> is trivial.  Should state sharing be desired, the state must be encapsulated in the derived instance of <strong>MynahSA::RequestResponseHandler</strong>.</li>
	<li>Each simultaneous connection to a server spawns a new thread.  Inside that thread, <strong>MynahSA::RequestResponseHandler::operator()</strong> is invoked.  Therefore, all code placed in implementations of <strong>operator()</strong> must be thread safe.  This is not a problem for <strong>PingRequestResponseHandler</strong> as it has no state.</li>
      </ul>

\section tut3constructors Constructor Objects

      Every time a new connection is created with MynahSA's TCP server a new instance of <strong>TCPArchiveStream</strong> is instantiated to provide archiving and serialization of the objects transmitted over the TCP connection.  As explained in \ref tutorial1 objects transmitted by shared pointer must provide a constructor object and the constructor object must be registered with all input archivers.  <strong>MynahSA</strong>'s network transports operate by transmitting objects by shared pointer, therefore, constructor functions must be provided for all objects that will be transmitted.  Constructor objects are created by deriving from base class <strong>MynahSA::StreamConstructor</strong>:

\code
class PingStreamConstructor : public MynahSA::StreamConstructor { 
public:
  PingStreamConstructor();

  virtual ~PingStreamConstructor();
  
  virtual void operator()(MynahSA::ArchiveStreamBase&;) const;
};
\endcode

      The implementation of operator() from examples/tutorial3/pingstreamconstructor.cpp, shown below, demonstrates the registration of <strong>RequestPing</strong> and <strong>ResponsePing</strong>: 
\code
using namespace MynahSA;

// build the template constructor classes for RequestPing and ResponsePing
MYNAHSA_BUILD_CONSTRUCTOR(RequestPing);
MYNAHSA_BUILD_CONSTRUCTOR(ResponsePing);

void PingStreamConstructor::operator()(ArchiveStreamBase& asb) const { 
  // Register RequestPing
  MYNAHSA_REGISTER_CONSTRUCTOR(asb, RequestPing);
  
  // Register ResponsePing
  MYNAHSA_REGISTER_CONSTRUCTOR(asb, ResponsePing);
}
\endcode
      The above implementation of operator() takes a bidirectional stream instance (class SSLArchiveStream or TCPArchiveStream), instantiates constructor objects and registers them with the stream.  A call to <strong>MYNAHSA_REGISTER_CONSTRUCTOR</strong> is required for each class transmitted by shared pointer across the network.<br><br>All of the components that are shared between client and server have now been demonstrated.  The next step is to pull the components together and build the server and client implementations.

\section tut3servimp Server Implementation

      The server is implemented in examples/tutorial3/tcpserver.cpp.  The steps in creating the server are:
      <ul>
	<li>Create a PingRequestResponseHandler.</li>
	<li>Create a PingStreamConstructor.</li>
	<li>Create a TCPConnectionManager, binding together the PingRequestResponseHandler and the stream constructor object.</li>
	<li>Create a TCPRPCServer, using the connection manager.</li>
	<li>Loop forever, waiting for connections.</li>
      </ul>
      We show the relevant code bits here from tcpserver.cpp - omitting exception handling and the calculation of the serverPort variable.

\code
    // initialize WinSock if on Win32 and SSL
    saInit();

...

    // step 1: build the ping request response handler
    PingRequestResponseHandler prh;
    
    // step 2: create the stream constructor object.
    PingStreamConstructor myConstructor;
  
    // step 4: instantiate a connection manager on the handler and the 
    //         stream constructor
    TCPConnectionManager cm(prh,
                            myConstructor);
                                              
    // step 5: build an ssl server on top of the connection manager instance
    TCPRPCServer server(cm, serverPort); 
    
    // step 6: enter loop where we serve requests forever
    while (1) { 
      try { 
        server.checkClients(5);
      } catch (const TCPServerConnectionError& sce) { 
        cerr << "Connection error detected: " << sce.what() << endl;
      }
    }
\endcode

\section tut3clientimp Client Implmenetation

      The client implementation is simple, and the steps are similar to the server.  The steps are:
      <ul>
	<li>Determine the IP Address and port of the server</li>
	<li>Create an instance of PingStreamConstructor</li>
	<li>Create the TCPRPCClient instance</li>
	<li>Call TCPRPCClient::rpc( ... ) with request objects and handle the response objects returned.</li>
      </ul>
      For brevity, the computation of the server's IP address and port are omitted.  Refer to examples/tutorial3/tcpclient.cpp for details.<br>The creation of a PingStreamConstructor is identical to the server.  The setup code looks like this:

\code
  PingStreamConstructor myConstructor;
  TCPRPCClient myClient(myConstructor, ina, port);\endcode
      The final step is to loop forever transmitting ping objects and receiving and handling responses.  The code for calculating the round trip time has been omitted:
      \code
while(1) {
  ...
  SHARED_PTR<RequestPing> rpreq(new RequestPing(t));
  // perform the remote procedure call
  SHARED_PTR<IoBase> resp(myClient.rpc(rpreq));
  if (resp->ioName() == ResponsePing().ioName()) {
    SHARED_PTR<ResponsePing> rsp = static_pointer_cast<ResponsePing>(resp);
    // Print round trip time
    ...
  }
}
\endcode

\section tut3summary Summary
      This tutorial has demonstrated the creation of a simple client/server using MynahSA's TCP transport mechanism.  The process of using TCP for creating a client/server system is reasonably simple, however, there are a few things to remember:
      <ul>
	<li>TCP is an insecure transport mechanism - meaning that all information is transmitted in the clear.  For applications used in-house this may be perfectly acceptable, but for Internet applications using the SSL transport is strongly recommended.</li>
	<li>There are only two differences between a client/server system and archiving to files: The server behavior is encapsulated in an object derived from <strong>MynahSA::RequestResponseHandler</strong> and a constructor object for <strong>MynahSA</strong>'s TCP transport is required.</li>
      </ul>

*/
