/************************************************************************************
 *    This file is part of the MynahSA streaming and archiving toolkit              *
 *    Copyright (C) 2006 Mynah-Software Ltd. All Rights Reserved.                   *
 *                                                                                  *
 *    This program is free software; you can redistribute it and/or modify          *
 *    it under the terms of the GNU General Public License, version 2               *
 *    as published by the Free Software Foundation.                                 *
 *                                                                                  *
 *    This program is distributed in the hope that it will be useful,               *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of                *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                 *
 *    GNU General Public License for more details.                                  *
 *                                                                                  *
 *    You should have received a copy of the GNU General Public License along       *
 *    with this program; if not, write to the Free Software Foundation, Inc.,       *
 *    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.                   *
 *                                                                                  *
 ************************************************************************************/

#include <iostream>

#include <mynahsa/mynahsa.hpp>

#include "pingrequestresponsehandler.hpp"
#include "pingstreamconstructor.hpp"

using namespace std;
using namespace MynahSA;

void usage() {
  cerr << "Usage: " << endl;
  cerr << " server  [-port <int>] [-cert <certificate file> -pkey <private key file>] [-CAfile <CA file>]" << endl;
  cerr << " where:" << endl;
  cerr << "   -port <int> specifies the port to operate on" << endl;
  cerr << "   -cert <certificate file> specifies the certificate file" << endl;
  cerr << "   -pkey <private key file> specifies the private key file" << endl;
  cerr << "   -CAfile <CA file> specify a certificate authority file. See" << endl;
  cerr << "    man SSL_CTX_load_verify_locations for information on preparing the CA file." << endl;
  cerr << " Note: both -cert and -pkey must be specified, either one alone is an error" << endl;
  cerr << "       Omitting both -cert and -pkey will cause the server to create a certificate" << endl;
  cerr << "       on the fly with default parameters." << endl;        
}

int main(int argc, char** argv) { 

  // initialize archiving and streaming library
  saInit();

  X509* cert = 0;                          // cert pointer
  EVP_PKEY* pkey = 0;                      // pkey pointer
  try { 
    // hard assign the server port for this example
    int serverPort = 7501;
    string certFile = "automatically generated";
    string privKeyFile = "automatically generated";
    string caFile = "";
    bool privKeySpecified = false;
    bool certSpecified = false;
    int i=1;
    while (i<argc) { 
      if (string(argv[i]) == "-port") { 
        i++;
        if (i >= argc) { 
          usage();
          return 1;
        }
        serverPort = StringCast<int>(argv[i]);
      } else if (string(argv[i]) == "-cert") { 
        i++;
        if (i >= argc) { 
          usage();
          return 1;
        }
        certSpecified = true;
        certFile = argv[i];
      } else if (string(argv[i]) == "-pkey") { 
        i++;
        if (i >= argc) { 
          usage();
          return 1;
        }
        privKeySpecified = true;
        privKeyFile = argv[i];
      } else if (string(argv[i]) == "-CAfile") { 
        i++;
        if (i >= argc) { 
          usage();
          return 1;
        }
        caFile = argv[i];
      } else { 
        usage();
        return 1;
      }
      i++;
    }

    if ((privKeySpecified && !certSpecified) ||
        (!privKeySpecified && certSpecified)) { 
      usage();
      return -1;
    }




    // step 1: build the ping request response handler
    PingRequestResponseHandler prh;
    
    // step 2: create the stream constructor object.
    PingStreamConstructor myConstructor;
    

    // Next - we fork the code path into two separate parts.  The first path is for
    //   strong client authentication.  The second allows anyone to connect.
    //
    // It is assumed that normal applications will not have two different code-paths
    //   but a strategic design decision will be taken ahead of time depending on the
    //   application authentication requirements.
    //
    //  Notes:
    //    For the secured code path, one must supply command line options:
    //       -cert <server certificate file>
    //       -pkey <server private key>
    //       -CAfile - the certificate authority file.
    //      Because a CAfile is required, clients will need to have their certificate 
    //        signed by the certificate authority before they will be allowed to 
    //        communicate with the server.  This method of authentication is especially
    //        useful for preventing unauthorized clients from accessing the server.
    //
    //    The unsecured code path creates a default certificate with some default parameters.
    //      The unsecured mechanism should only be used where the security of the data in transit
    //      is required, but no authentication of either party is necessary.  Think long and hard
    //      before using the un-authenticated approach!
    
    if (privKeySpecified) {
      // step 3: create a connection verification object
      // certificate standard identifier - forces client certificate CN to be identical to the 
      // name of the client which is obtained from name lookup!
      SSLConnectionCertVerifier myVerifier;
    
      // step 4: instantiate a connection manager on the handler and the stream constructor
      SSLConnectionManager cm(prh,
                              myConstructor,
                              myVerifier); 
                                                
      // step 5: build an ssl server on top of the connection manager instance
      SSLRPCServer server(cm, certFile, privKeyFile, serverPort,caFile);
      
      // step 6: enter loop where we serve requests forever
      while (1) { 
        try { 
          server.checkClients(5);
        } catch (const ServerConnectionException& sce) { 
          cerr << "Connection exception detected: " << sce.what() << endl;
        }
      }
    } else { 
      // step 3: create a connection verification object
      // Default verifier - accepts everything
      SSLConnectionVerifier myVerifier;
      
      // step 4: instantiate a connection manager on the handler and the stream constructor
      SSLConnectionManager cm(prh,
                              myConstructor,
                              myVerifier); 
                                                
      // step 5: build an ssl server on top of the connection manager instance
      mkcert(&cert, &pkey, 1024, 0, 760);      // create cert and pkey - completely self signed 1024 bit
      SSLRPCServer server(cm, cert, pkey, serverPort,caFile); 
      
      // step 6: enter loop where we serve requests forever
      while (1) { 
        try { 
          server.checkClients(5);
        } catch (const ServerConnectionException& sce) { 
          cerr << "Connection error detected: " << sce.what() << endl;
        }
      }      
    }
  } catch (const ServerException& se) { 
    cerr << "Fatal server error occurred: " << se.what() << endl;
  }
  // cleanup on error
  if (pkey != 0) {
    EVP_PKEY_free(pkey);
  }
  
  if (cert != 0) { 
    X509_free(cert);
  }
  
}
