/************************************************************************************
 *    This file is part of the MynahSA streaming and archiving toolkit              *
 *    Copyright (C) 2006 Mynah-Software Ltd. All Rights Reserved.                   *
 *                                                                                  *
 *    This program is free software; you can redistribute it and/or modify          *
 *    it under the terms of the GNU General Public License, version 2               *
 *    as published by the Free Software Foundation.                                 *
 *                                                                                  *
 *    This program is distributed in the hope that it will be useful,               *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of                *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                 *
 *    GNU General Public License for more details.                                  *
 *                                                                                  *
 *    You should have received a copy of the GNU General Public License along       *
 *    with this program; if not, write to the Free Software Foundation, Inc.,       *
 *    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.                   *
 *                                                                                  *
 ************************************************************************************/

#include <iostream>

#include "personrecord.hpp"
#include "persondatabase.hpp"

#include <mynahsa/iarchive.hpp>
#include <mynahsa/oarchive.hpp>

#include <mynahsa/itextstream.hpp>
#include <mynahsa/otextstream.hpp>

#include <mynahsa/ibinarystream.hpp>
#include <mynahsa/obinarystream.hpp>


#include <mynahsa/iarchiveregister.hpp>

#include <fstream>

#include <map>

#include <mynahsa/spimpl.hpp>


using namespace std;
using namespace MynahSA;


MYNAHSA_BUILD_CONSTRUCTOR( ExtendedPersonRecord );

int main() {

  cout << "Tutorial 1 ..." << endl;

  { 
    PersonRecord testRecord("Brett",32,PersonRecord::Blue);
  
    cout << "Person record contains: " << endl << testRecord << endl;

   // create a text formatted output stream
    OTextStream ots(cout);
  
    // create an output archive on the formatted stream
    OArchive<OTextStream> oa(ots);

    // store the object onto the stream  - in this
    //  case, the archive contents will be placed onto
    //  cout
    cout << "testRecord on stream looks like this: " << endl;
    oa << testRecord;
    cout << endl;
  
    cout << "testRecord prior to saving: " << endl << testRecord << endl;
  }
  
  cout << endl;
  cout << "Storing and loading a record on binary file streams: "<< endl;
  cout << endl;
  
  {
    PersonRecord testRecord("Brett",32,PersonRecord::Blue);
  
    ofstream ofs("testsave.dat");
    OBinaryStream obs(ofs);
    OArchive<OBinaryStream> obas(obs);
    obas << testRecord;
    
    // close the stream - force everything to be written
    // ofs is implicitly closed and flushed on scope exit
  }
  // now recover the data from the file
  
  {
    ifstream ifs("testsave.dat");
    IBinaryStream ibs(ifs);
    IArchive<IBinaryStream> ibas(ibs);  // input binary archive stream
  
    PersonRecord testRecord2;
    ibas >> testRecord2;
    cout << "testRecord2 after restore from file is: " << endl << testRecord2 << endl;
  }  
  
  cout << endl;
  cout << "STL test - stream a multimap onto a file, and recover it: "<< endl;
  cout << endl;
  
  // STL part - create a multimap of PersonRecords, and store them
  {

  
    //BMS20060630: All .fisrt members of inserters made const for SUNpro

    multimap<int, PersonRecord> mm;
    PersonRecord brett("Brett",32,PersonRecord::Blue);
    mm.insert(pair<const int, PersonRecord>(32,brett));
    
    PersonRecord agnes("Agnes",35,PersonRecord::Blue);
    mm.insert(pair<const int, PersonRecord>(35, agnes));
    
    PersonRecord john("John",32,PersonRecord::Brown);
    mm.insert(pair<const int, PersonRecord>(32,john));
    
    
    // now drop mm onto a file stream 
    ofstream ofs("testmultimap.dat");
    OBinaryStream obs(ofs);
    OArchive<OBinaryStream> obas(obs);
    obas << mm;
  }
  
  {  // now recover the multimap off of the stream and proceed to print a specific range of data
    ifstream ifs("testmultimap.dat");
    IBinaryStream ibs(ifs);
    IArchive<IBinaryStream> ibas(ibs);
     
    multimap<int, PersonRecord> mm;
    ibas >> mm;
        
    cout << "People between the age of 30 and 40: " << endl;
    for (multimap<int, PersonRecord>::const_iterator cit = mm.lower_bound(30);
         cit != mm.upper_bound(40);
         ++cit) { 
      cout << (*cit).second << endl;
    }
    
  }
  
  
  // Final part on graph like data structures.
  {
    // create the database
    PersonDatabase db;
    // create the individuals
    SHARED_PTR<ExtendedPersonRecord> brett(new ExtendedPersonRecord("Brett",32,PersonRecord::Blue));
    SHARED_PTR<ExtendedPersonRecord> agi(new ExtendedPersonRecord("Agi",35,PersonRecord::Blue));
    SHARED_PTR<ExtendedPersonRecord> jim(new ExtendedPersonRecord("Jim",65, PersonRecord::Blue));
    
    // construct the friendship graph
    brett->addFriend(agi);
    brett->addFriend(jim);
    
    agi->addFriend(brett);
    agi->addFriend(jim);
    
    jim->addFriend(brett);
    jim->addFriend(agi);
    
    // add everything to the database
    db.insertPerson(brett);
    db.insertPerson(agi);
    db.insertPerson(jim);
    
    // now db holds everything - lets see what this looks like on the stream:
    cout << "Writing archive onto disk: persondb.txt ...";
    ofstream ofs("persondb.txt");
    OTextStream ots(ofs);
    OArchive<OTextStream> otas(ots);
    otas << db;
    cout << "done" << endl;
  }
 
  {

    cerr << "Reading archive into database: persondb.txt ...";
    // create the input stream, bind it to the data file
    ifstream ifs("persondb.txt");
    // create the formatted input text stream
    ITextStream its(ifs);
    // create the input archiver instance
    IArchive<ITextStream> itas(its);

    // Register the construction object with the IArchive instance
    MYNAHSA_REGISTER_CONSTRUCTOR(itas, ExtendedPersonRecord);
    
    PersonDatabase db;
    itas >> db;
    
    cout << "done" << endl;
    cout << endl << "Here is the contents of the database: " << endl << endl;
    cout << db << endl;
  }
 
 
 
 
  return 0;
}

