/************************************************************************************
 *    This file is part of the MynahSA streaming and archiving toolkit              *
 *    Copyright (C) 2006 Mynah-Software Ltd. All Rights Reserved.                   *
 *                                                                                  *
 *    This program is free software; you can redistribute it and/or modify          *
 *    it under the terms of the GNU General Public License, version 2               *
 *    as published by the Free Software Foundation.                                 *
 *                                                                                  *
 *    This program is distributed in the hope that it will be useful,               *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of                *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                 *
 *    GNU General Public License for more details.                                  *
 *                                                                                  *
 *    You should have received a copy of the GNU General Public License along       *
 *    with this program; if not, write to the Free Software Foundation, Inc.,       *
 *    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.                   *
 *                                                                                  *
 ************************************************************************************/

#include "persondatabase.hpp"
  
using namespace std;
using namespace MynahSA;
  
std::string PersonDatabase::ioName() const { 
  return "PersonDatabase";
}
  
void PersonDatabase::serialize(Archive& ar) {
  ar & _ageMap;
  ar & _nameMap;
  ar & _eyeColorMap;
}
  
PersonDatabase::PersonDatabase() {
}
  
PersonDatabase::~PersonDatabase() {
}
  
void PersonDatabase::insertPerson(spRecord spr) {
  //BMS20060630: made inserter take a const for first parameter for compatibility with SUNPro compiler
  _ageMap.insert(pair<const int, spRecord>(spr->_age,spr));
  _nameMap.insert(pair<const string, spRecord>(spr->_name, spr));
  _eyeColorMap.insert(pair<const PersonRecord::EyeColor, spRecord>(spr->_eyeColor, spr));
}
  
void PersonDatabase::removePerson(spRecord r) {
  for (multimap<int, spRecord>::iterator mit = _ageMap.lower_bound(r->_age);
       mit != _ageMap.upper_bound(r->_age);
       ++mit) { 
    if ( (*mit).second == r) { 
      _ageMap.erase(mit);
      break;
    }
  }
  
  for (multimap<string, spRecord>::iterator mit = _nameMap.lower_bound(r->_name);
       mit != _nameMap.upper_bound(r->_name);
       ++mit) { 
    if ( (*mit).second == r) { 
      _nameMap.erase(mit);
      break;
    }
  }
   
  for (multimap<PersonRecord::EyeColor, spRecord>::iterator mit = _eyeColorMap.lower_bound(r->_eyeColor);
       mit != _eyeColorMap.upper_bound(r->_eyeColor);
       ++mit) { 
    if ( (*mit).second == r) { 
      _eyeColorMap.erase(mit);
      break;
    }
  }
}

PersonDatabase::spHitList PersonDatabase::getAllRecords() const { 
  spHitList hitList;
  for (multimap<int, spRecord>::const_iterator cit = _ageMap.begin();
       cit != _ageMap.end();
       ++cit) {
    hitList.push_back( (*cit).second );
  }
  return hitList;
}
  
PersonDatabase::spHitList PersonDatabase::getRecordsByAge(int low, int high) const {

  spHitList hitList;

  multimap<int, spRecord>::const_iterator start = _ageMap.lower_bound(low);
  multimap<int, spRecord>::const_iterator end   = _ageMap.upper_bound(high);
  for (multimap<int, spRecord>::const_iterator cit = start;
       cit != end;
       ++cit) { 
    hitList.push_back( (*cit).second );     
  }
  return hitList;
}

PersonDatabase::spHitList PersonDatabase::getRecordsByName(std::string name) const {
  spHitList hitList;
  multimap<string, spRecord>::const_iterator start = _nameMap.lower_bound(name);
  multimap<string, spRecord>::const_iterator end   = _nameMap.upper_bound(name);
  for (multimap<string, spRecord>::const_iterator cit = start;
       cit != end;
       ++cit) { 
    hitList.push_back( (*cit).second );     
  }
  
  return hitList;
}
  
PersonDatabase::spHitList PersonDatabase::getRecordsByEyeColor(PersonRecord::EyeColor ec) const {
  spHitList hitList;
  multimap<PersonRecord::EyeColor, spRecord>::const_iterator start = _eyeColorMap.lower_bound(ec);
  multimap<PersonRecord::EyeColor, spRecord>::const_iterator end   = _eyeColorMap.upper_bound(ec);
  for (multimap<PersonRecord::EyeColor, spRecord>::const_iterator cit = start;
       cit != end;
       ++cit) { 
    hitList.push_back( (*cit).second );     
  }
  
  return hitList;
}


ostream& operator<<(ostream& os, const PersonDatabase& pd) { 
  PersonDatabase::spHitList allRecords = pd.getAllRecords();
  for (PersonDatabase::spHitList::const_iterator cit(allRecords.begin());
       cit != allRecords.end();
       ++cit) {
    const PersonRecord& pr(*(STATIC_PTR_CAST<PersonRecord>(*cit))); // cast to a person record
    os << pr;  // use its stream operator
    os << "Friends: ";
    const list< SHARED_PTR< ExtendedPersonRecord > >& friends( (*cit)->getFriends());
    for (list< SHARED_PTR<ExtendedPersonRecord> >::const_iterator lit(friends.begin());
         lit != friends.end();
         ++lit) { 
      os << (*lit)->_name << " ";
    }
    os << endl;
    os << endl;
  }
  return os;
}

