/************************************************************************************
 *    This file is part of the MynahSA streaming and archiving toolkit              *
 *    Copyright (C) 2006 Mynah-Software Ltd. All Rights Reserved.                   *
 *                                                                                  *
 *    This program is free software; you can redistribute it and/or modify          *
 *    it under the terms of the GNU General Public License, version 2               *
 *    as published by the Free Software Foundation.                                 *
 *                                                                                  *
 *    This program is distributed in the hope that it will be useful,               *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of                *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                 *
 *    GNU General Public License for more details.                                  *
 *                                                                                  *
 *    You should have received a copy of the GNU General Public License along       *
 *    with this program; if not, write to the Free Software Foundation, Inc.,       *
 *    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.                   *
 *                                                                                  *
 ************************************************************************************/

#ifndef __person_database_hpp
#define __person_database_hpp

#include <mynahsa/spimpl.hpp>


#include <vector>
#include <map>

#include <mynahsa/iobase.hpp>
#include <mynahsa/archive.hpp>

#include "extendedpersonrecord.hpp"


class PersonDatabase : public MynahSA::IoBase {
public:
  //! typedef to reduce typing - spRecord is a shared pointer to an extended person record
  typedef SHARED_PTR<ExtendedPersonRecord> spRecord;
  
  //! a hitlist is a list of shared pointers to records.
  typedef std::list<spRecord> spHitList;
  
  //! IoBase reimplementation
  std::string ioName() const;  // return the io name
  
  //! IoBase reimplementation
  void serialize(MynahSA::Archive&);
  
  //! constructor
  PersonDatabase();
  
  //! destructor
  ~PersonDatabase();
  
  //! insert a person into the database
  void insertPerson(spRecord);
  
  //! remove a person from the database
  void removePerson(spRecord);
  
  //! retrieve all records
  spHitList getAllRecords() const;
  
  //! get records by age-range
  spHitList getRecordsByAge(int low, int high) const;
  
  //! get records by name
  spHitList getRecordsByName(std::string) const;
  
  //! get records by eye-color
  spHitList getRecordsByEyeColor(PersonRecord::EyeColor) const;
  
  
private:

  //! container indexed by age
  std::multimap<int, spRecord > _ageMap;
  
  //! container indexed by 
  std::multimap<std::string, spRecord > _nameMap;

  //! container indexed by eye color
  std::multimap<PersonRecord::EyeColor, spRecord > _eyeColorMap;

};

// output stream operator - write the whole database to a output stream
std::ostream& operator<<(std::ostream& os, const PersonDatabase& pd);


#endif
