/************************************************************************************
 *    This file is part of the MynahSA streaming and archiving toolkit              *
 *    Copyright (C) 2006 Mynah-Software Ltd. All Rights Reserved.                   *
 *                                                                                  *
 *    This program is free software; you can redistribute it and/or modify          *
 *    it under the terms of the GNU General Public License, version 2               *
 *    as published by the Free Software Foundation.                                 *
 *                                                                                  *
 *    This program is distributed in the hope that it will be useful,               *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of                *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                 *
 *    GNU General Public License for more details.                                  *
 *                                                                                  *
 *    You should have received a copy of the GNU General Public License along       *
 *    with this program; if not, write to the Free Software Foundation, Inc.,       *
 *    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.                   *
 *                                                                                  *
 ************************************************************************************/

#include <iostream>
#include <string>

#include <mynahsa/spimpl.hpp>
#include <mynahsa/stringcast.hpp>


#include <mynahsa/tcprpcclient.hpp>

#include "pingstreamconstructor.hpp"

#include <mynahsa/tcparchivestream.hpp>

#include <mynahsa/clientexception.hpp>

#include <mynahsa/sainit.hpp>


#ifndef WIN32
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <sys/time.h>
#include <unistd.h>
#else
#include <windows.h>
#include <winsock2.h>
#endif

#include <time.h>




using namespace std;
using namespace MynahSA;

/** This function returns a "private time".  Which roughly means that you shouldn't look too closely
 *  into what is being returned, rather interrogate it with the function microsecondInterval below.
 *  The reason for this is that accurate "time of day" timing seems non-existent on Win32.  On Win32,
 *  we use the processor tick counter, which varies in frequency depending on the CPU clock rate.  
 *  
 *  On unix, the value is just the microseconds since the epoc, using the unistd api gettimeofday.
 */
unsigned long long getCurrentTime() { 
  unsigned long long privTime = 0;
#ifdef WIN32
  QueryPerformanceCounter((LARGE_INTEGER*) &privTime);
#else
  struct timeval tv;
  gettimeofday(&tv,0);
  privTime = tv.tv_sec* 1000000 + tv.tv_usec;  // mutliply seconds by 1e6, add useconds

#endif
  return privTime;
}

//! compute the number of elapsed microseconds between two calls to getCurrentTime
unsigned long microsecondInterval(unsigned long long a, unsigned long long b) { 
  // working in private times.  Microseconds on unix, something else (uP cycles) on Windows
#ifdef WIN32
  // Note: this throws away a lot of precision.  It is not meant to be an accurate timer!

  unsigned long long interval = b - a;  // should be a small number, but remember 1 second on a fast pIV will exceed 2^32
  // convert ticks / second into ticks per microsecond
  unsigned long long value;
  QueryPerformanceFrequency((LARGE_INTEGER*) &value);
  value /= 1000000;
  return interval / value;
#else
  return b - a;
#endif
};



//! print program command line options and return
void usage() { 
  
  cerr << "Usage: tcpclient <machine name> <port number>" << endl;
  cerr << "       <ip address/ fully qualified machine name> is the target machine's name, running server." << endl;
  cerr << "       <port number> is the port number that the server is operating on." << endl;
  
}

int main(int argc, char** argv) { 

  saInit();

  try { 
  
    if (argc != 3) {
      usage();
      return 1;
    }

    string machineName = argv[1];
    int port= StringCast<int>(argv[2]);
    
    cerr << "Performing setup" << endl;

    // create the constructor object.
    PingStreamConstructor myConstructor;

    TCPRPCClient myClient(myConstructor, machineName, port);
    while (1) {
      // create the request object
      SHARED_PTR<RequestPing> rpreq(new RequestPing(getCurrentTime()));

      // perform the remote procedure call
      SHARED_PTR<IoBase> resp(myClient.rpc(rpreq));
    
      // interpret the response
      if (resp->ioName() == "ResponsePing") {
        SHARED_PTR<ResponsePing> rsp = STATIC_PTR_CAST<ResponsePing>(resp);
        cerr << "ResponsePing isOk is: " << rsp->isOk() << " ";
        // now use the response ping's seconds and microseconds objects to compute the real-time that has passed
        // so we can compute the latency involved in the comms process

        long long diff = microsecondInterval(rsp->getUSeconds(), getCurrentTime());
        
        cerr << "Round trip time: " << (((double) diff)*1e-6) << " seconds" << endl;
        
      } 
#ifndef WIN32                         
      sleep(1);  // sleep for one second.
#else
      Sleep(1000);
#endif
    }
  } catch (StringCastException& bc) { 
    cerr << "Port number could not be converted into an integer.  Check arguments!" << endl;
    return 1;
  } catch (ExceptionBase& eb) { 
    cerr << "Caught a fatal exception: " << eb.what() << endl;
  } catch (...) { 
    cerr << "Uncaught exception.  Check and debug!" << endl;
  }

}
