/************************************************************************************
 *    This file is part of the MynahSA streaming and archiving toolkit              *
 *    Copyright (C) 2006 Mynah-Software Ltd. All Rights Reserved.                   *
 *                                                                                  *
 *    This program is free software; you can redistribute it and/or modify          *
 *    it under the terms of the GNU General Public License, version 2               *
 *    as published by the Free Software Foundation.                                 *
 *                                                                                  *
 *    This program is distributed in the hope that it will be useful,               *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of                *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                 *
 *    GNU General Public License for more details.                                  *
 *                                                                                  *
 *    You should have received a copy of the GNU General Public License along       *
 *    with this program; if not, write to the Free Software Foundation, Inc.,       *
 *    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.                   *
 *                                                                                  *
 ************************************************************************************/

#include "database.hpp"

#include "dbregister.hpp"

#include <fstream>
#include <mynahsa/ibinarystream.hpp>
#include <mynahsa/obinarystream.hpp>
#include <mynahsa/iarchive.hpp>
#include <mynahsa/oarchive.hpp>


// local define for locking of procedures
// Each procedure that may compete with other threads needs a call to LOCK to prevent problems!
#define LOCK Lock __l(_mutex)


using namespace std;
using namespace MynahSA;

Database::Database() : _nextUID(1) { 
}

Database::Database(const string& fileName) : _fileName(fileName), _nextUID(1) { 
  ifstream ifs(fileName.c_str());  // open input file
  // only load if stream is open
  if (ifs.is_open()) {
    IBinaryStream ibs(ifs);
    IArchive<IBinaryStream> iabs(ibs);
    DBRegister myRegister;
    myRegister(iabs);
    
    // restore contents
    iabs >> (*this);
  }
  
}

Database::~Database() {
  // write the database to disk
  if (_fileName != "") { 
    ofstream ofs(_fileName.c_str());
    OBinaryStream obs(ofs);
    OArchive<OBinaryStream> oabs(obs);
    oabs << (*this);
  }
}

string Database::ioName() const { 
  return "Database";
}

void Database::serialize(Archive& ar) { 
  // lock the database!
  LOCK;
  
  ar & _uidToEPR;
  ar & _eprToUID;
  ar & _nextUID;
  ar & _ageMap;
  ar & _nameMap;
  ar & _eyeColorMap;
  
  // note: none of the other members are saved!
  // _fileName may be changed on a per-instance basis, and
  // _mutex is specific to the state of the application and running threads.
}

DBResult Database::queryName(const string& nameStart, const string& nameEnd) const { 
  LOCK;

  DBHitList hl;

  string nameEndQuery;
  if (nameEnd == "") { 
    nameEndQuery = nameStart;
  } else {
    nameEndQuery = nameEnd;
  }
  
  basicQuery(nameStart, nameEndQuery, _nameMap, hl);
  
  return DBResult(DBResult::Ok, "", hl);
}

DBResult Database::queryAge(int ageStart, int ageEnd) const { 
  LOCK;
  
  DBHitList hl;

  basicQuery(ageStart, ageEnd, _ageMap, hl);
  
  return DBResult(DBResult::Ok, "", hl);
}

DBResult Database::queryEyeColor(PersonRecord::EyeColor ec) const { 
  LOCK;
  
  DBHitList hl;
  basicQuery(ec, ec, _eyeColorMap, hl);
  return DBResult(DBResult::Ok, "", hl);
}

DBResult Database::queryAll() const { 
  LOCK;
  
  DBHitList hl;
  for (map<unsigned int, spRecord>::const_iterator cit = _uidToEPR.begin();
       cit != _uidToEPR.end();
       ++cit) {
    hl.push_back(DBHit( (*cit).first, (*cit).second ) );
  }
  return DBResult(DBResult::Ok, "", hl);
}


DBResult Database::modifyRecord(unsigned int uid, spRecord newRecord) { 
  LOCK;
  
  map<unsigned int, spRecord>::iterator mit =  _uidToEPR.find(uid);
  if (mit == _uidToEPR.end()) {
    return DBResult(DBResult::Error, "Database::modifyRecord - unknown unique record identifier", DBHitList());
  } else { 
    
    // update the record, preserving the UID
    removeSPRecord( (*mit).second, _nameMap );
    removeSPRecord( (*mit).second, _ageMap );
    removeSPRecord( (*mit).second, _eyeColorMap );
    
    // now change the contents of spRecord
    (*(*mit).second) = (*newRecord);  // this copies the old record on top of the new one.
    
    // now re-insert the record into _nameMap, _ageMap and _eyeColorMap
    
    _nameMap.insert    (pair<const string,                 spRecord>(newRecord->_name,     (*mit).second));
    _ageMap.insert     (pair<const int,                    spRecord>(newRecord->_age,      (*mit).second));
    _eyeColorMap.insert(pair<const PersonRecord::EyeColor, spRecord>(newRecord->_eyeColor, (*mit).second));
    DBHitList hl;
    hl.push_back(DBHit( uid, (*mit).second ) );
    return DBResult(DBResult::Ok, "", hl);
  }

}

DBResult Database::deleteRecord(unsigned int uid) { 
  LOCK;
  
  map<unsigned int, spRecord>::iterator mit =  _uidToEPR.find(uid);
  if (mit == _uidToEPR.end()) {
    return DBResult(DBResult::Error, "Database::deleteRecord - unknown unique record identifier", DBHitList());
  } else { 
    removeSPRecord( (*mit).second, _nameMap );
    removeSPRecord( (*mit).second, _ageMap );
    removeSPRecord( (*mit).second, _eyeColorMap );
    
    map<spRecord, unsigned int>::iterator mit2 = _eprToUID.find((*mit).second);
    _eprToUID.erase(mit2);
    
    _uidToEPR.erase(mit);
    return DBResult(DBResult::Ok, "", DBHitList());
  }
}

DBResult Database::insertRecord(spRecord newRecord) {

  // assign UID and increment uid count
  unsigned int uid = _nextUID++;
  
  // insert into database
  _uidToEPR[uid] = newRecord;
  _eprToUID[newRecord] = uid;
 
  // and update query maps
  _nameMap.insert    (pair<const string,                 spRecord>(newRecord->_name,     newRecord));
  _ageMap.insert     (pair<const int,                    spRecord>(newRecord->_age,      newRecord));
  _eyeColorMap.insert(pair<const PersonRecord::EyeColor, spRecord>(newRecord->_eyeColor, newRecord));

  // and return the results
  DBHitList hl;
  hl.push_back(DBHit(uid, newRecord));
  
  return DBResult(DBResult::Ok, "", hl);
}

#undef LOCK
