/************************************************************************************
 *    This file is part of the MynahSA streaming and archiving toolkit              *
 *    Copyright (C) 2006 Mynah-Software Ltd. All Rights Reserved.                   *
 *                                                                                  *
 *    This program is free software; you can redistribute it and/or modify          *
 *    it under the terms of the GNU General Public License, version 2               *
 *    as published by the Free Software Foundation.                                 *
 *                                                                                  *
 *    This program is distributed in the hope that it will be useful,               *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of                *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                 *
 *    GNU General Public License for more details.                                  *
 *                                                                                  *
 *    You should have received a copy of the GNU General Public License along       *
 *    with this program; if not, write to the Free Software Foundation, Inc.,       *
 *    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.                   *
 *                                                                                  *
 ************************************************************************************/

#ifndef __database_hpp
#define __database_hpp

#include <mynahsa/archive.hpp>
#include <map>
#include <string>
#include <utility>

#include "dbtypedef.hpp"
#include "dbresult.hpp"


#include <mynahsa/thread.hpp>
class Database : public MynahSA::IoBase { 
public:

  /**
   * Constructor
   * @return  - nothing
   */
  Database();
  
  /**
   * File constructor - attempt to recreate a database from a file
   * @param filename - nothing
   * @return 
   */
  Database(const std::string& filename);
  
  /**
   * Destructor - writes file to filename, if filename is set
   * @return 
   */
  ~Database();

  // reimplementation from base class MynahSA::IoBase
  //! return the ioName - "Database"
  std::string ioName() const;
  
  //! serialization of this database
  void serialize(MynahSA::Archive& ar);


  // Database methods

  /**
   * Query by name - two names are provided, a start name and an end name.  All names in within 
   * the alphabetical range will be returned on the hitlist.
   * @param nameStart - the first name (start query by)
   * @param nameEnd  - the last name (if empty, query searches for only entries matching firstName)
   * @return - DBResult, the results of the search
   */
  DBResult queryName(const std::string& nameStart, const std::string& nameEnd) const;
  
  /**
   * Query by age range. Prerequisite: ageStart <= ageEnd
   * @param ageStart - the lowest age allowed to be returned on the hitlist
   * @param ageEnd  - the highest age allowed to be returned on the hitlist
   * @return - DBResult, the results of the search
   */
  DBResult queryAge(int ageStart, int ageEnd) const;
  
  /**
   * Query by eye color - returns all records with matching eye color.
   * @param ec - the specific eye color
   * @return - DBResult, the results of the search
   */
  DBResult queryEyeColor(PersonRecord::EyeColor ec) const;

  /**
   * Return all records in the database
   * @return - DBResult, hitlist contains all records in this case
   */
  DBResult queryAll() const;

  /**
   * Modify a record by unique identifier.  This method replaces the ExtendedPersonRecord identified
   * by uid with a new record, provided by the caller.
   * @param uid - Unique identifier for the record to be replaced
   * @param newRecord - the replacement record
   * @return 
   */
  DBResult modifyRecord(unsigned int uid, spRecord newRecord);

  /**
   * Delete a record by unique identifier
   * @param uid 
   * @return - DBResult, indicates Ok for success, otherwise an error and associated string.
   */
  DBResult deleteRecord(unsigned int uid);

  /**
   * Insert a record into the database.
   * @param newRecord - The new record to be inserted
   * @return - the result of the record hit.  The HitList contains the newly inserted record
   *            and associated unique id.
   */
  DBResult insertRecord(spRecord newRecord);

private:

  
  /**
   * basicQuery is a templated implementation of the query operation for all three types of queries
   * described above.
   * @param start - the lowest allowed value in the search
   * @param end - the highest allowed value in the search
   * @param sourceMap - the map to take the data from
   * @param hl - the hit list to return
   */
  template<class T>
  void basicQuery(const T& start, 
                  const T& end,
                  const std::multimap<T, spRecord>& sourceMap,
                  DBHitList& hl) const { 
    hl.clear();
    typename std::multimap<T, spRecord>::const_iterator startIt;
    typename std::multimap<T, spRecord>::const_iterator endIt;
  
    // force correct ordering for iterators!
    if (start <= end) { 
      startIt = sourceMap.lower_bound(start);
      endIt = sourceMap.upper_bound(end);
    } else { 
      startIt = sourceMap.lower_bound(end);
      endIt = sourceMap.upper_bound(start);
    }
    for (typename std::multimap<T, spRecord>::const_iterator mit = startIt;
         mit != endIt;
         ++mit) { 
      spRecord localRec = (*mit).second;
      typename std::map<spRecord, unsigned int>::const_iterator cmit = _eprToUID.find(localRec);
      // by definition, cmit must be a valid iterator.  If it is not there is a major bug with insertRecord!
      unsigned int uid = cmit->second;
      hl.push_back( DBHit(uid, localRec) );
    }                           
  }                               

  

  /**
   * Remove an instance from a map.  Preforms map lookup and removal of reference.
   * @param record - the record to remove
   * @param map - the map containing the instance
   */
  template<class T>
  void removeSPRecord(spRecord record, std::multimap<T, spRecord>& map) {
    for (typename std::multimap<T, spRecord>::iterator mit = map.begin();
         mit != map.end();
         ++mit) { 
      if ( (*mit).second == record) { 
        // remove it
        map.erase( mit );
        return;
      }
    }
  }
    

  // data container stuff - needed for safe keeping of data

  //! unique map from Id to pointer to extended person record
  std::map< unsigned int, spRecord > _uidToEPR;
  
  //! unique map from extended person record to UID
  std::map< spRecord, unsigned int > _eprToUID;

  //! the next UID to assign, when a new record is added
  unsigned int _nextUID;


  // now stuff for querying

  //! map age against spRecord
  std::multimap<int, spRecord > _ageMap;
  
  //! map name against spRecord
  std::multimap<std::string, spRecord > _nameMap;
  
  //! map Eye Color against spRecord
  std::multimap<PersonRecord::EyeColor, spRecord > _eyeColorMap;

  // Non archived members:
  //! the filename to read/write the database from
  std::string _fileName; 

  
  /** lock is critical - since multiple threads will be entering database's methods we must make them
   *  thread safe!.
   */
  mutable MynahSA::Mutex _mutex;
};



#endif
