/************************************************************************************
 *    This file is part of the MynahSA streaming and archiving toolkit              *
 *    Copyright (C) 2006 Mynah-Software Ltd. All Rights Reserved.                   *
 *                                                                                  *
 *    This program is free software; you can redistribute it and/or modify          *
 *    it under the terms of the GNU General Public License, version 2               *
 *    as published by the Free Software Foundation.                                 *
 *                                                                                  *
 *    This program is distributed in the hope that it will be useful,               *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of                *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                 *
 *    GNU General Public License for more details.                                  *
 *                                                                                  *
 *    You should have received a copy of the GNU General Public License along       *
 *    with this program; if not, write to the Free Software Foundation, Inc.,       *
 *    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.                   *
 *                                                                                  *
 ************************************************************************************/

#ifndef __database_driver_hpp
#define __database_driver_hpp


#include "dbresult.hpp"
#include "dbtypedef.hpp"
#include "personrecord.hpp"

/** class DatabaseDriver is a pure base class designed to take and wrap up an interface to class
 *  database.  The idea is that the command parser will call methods on this pure base class as
 *  they are received from the keyboard or other input mechanism.  The derived instances of this 
 *  class will invoke commands through MynahSA's RPC mechanism or alternatively through a direct 
 *  instance of class database.
 */
class DatabaseDriver { 
public:
  /**
   * Constructor
   * @return Nothing
   */
  DatabaseDriver();
  /**
   * Destructor
   * @return Nothing
   */
  virtual ~DatabaseDriver();
  
  /**
   * Perform a queryName() operation on the database
   * @param nameStart - the lowest string value name for acceptable hits
   * @param nameEnd - the highest string value name for acceptable hits
   * @return - the results of the query
   */
  virtual DBResult queryName(const std::string& nameStart, const std::string& nameEnd) const = 0;
  
  /**
   * Perform a queryAge query on the database implementation.
   * @param ageStart - the lowest acceptable age to be returned as a hit
   * @param ageEnd - the highest acceptable age to be returned as a hit
   * @return - the results of the query
   */
  virtual DBResult queryAge(int ageStart, int ageEnd) const = 0;
  
  /**
   * Perform a query on eye color.
   * @param ec - EyeColor for hits.  Only hits matching this eye color will be returned.
   * @return - the results of the query
   */
  virtual DBResult queryEyeColor(PersonRecord::EyeColor ec) const = 0;
  
  /**
   * Get all database records
   * @return - all records
   */
  virtual DBResult queryAll() const = 0;
  
  /**
   * Modify a record - this method modfies a record in the database.  Specifically, it
   * replaces the name, age and eye color fields of a record with the parameters provided in
   * calling this routine.
   * @param uid - the unique identifier of the record to modify
   * @param name - the replacement name for the record
   * @param age - the replacement age for the record
   * @param ec - the replacement eye color for the record
   * @return - the results from the operation
   */
  virtual DBResult modifyRecord(unsigned int uid, 
                                const std::string& name,
                                int age,
                                PersonRecord::EyeColor ec) = 0;
                                
  /**
   * Delete a record from the database
   * @param uid - the unique identifer for the record to be removed.
   * @return - the results from the operation
   */
  virtual DBResult deleteRecord(unsigned int uid)=0;
  
  /**
   * Insert a new record into the database.  
   * @param name - the name of the new person to be added to the database
   * @param age - the age of the person to be added
   * @param ec - the eye color of the person to be added
   * @return - The result contains one hit - the new record, and its unique identifier
   */
  virtual DBResult insertRecord(const std::string& name,
                                int age,
                                PersonRecord::EyeColor ec) = 0;
};


#endif
