/************************************************************************************
 *    This file is part of the MynahSA streaming and archiving toolkit              *
 *    Copyright (C) 2006 Mynah-Software Ltd. All Rights Reserved.                   *
 *                                                                                  *
 *    This program is free software; you can redistribute it and/or modify          *
 *    it under the terms of the GNU General Public License, version 2               *
 *    as published by the Free Software Foundation.                                 *
 *                                                                                  *
 *    This program is distributed in the hope that it will be useful,               *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of                *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                 *
 *    GNU General Public License for more details.                                  *
 *                                                                                  *
 *    You should have received a copy of the GNU General Public License along       *
 *    with this program; if not, write to the Free Software Foundation, Inc.,       *
 *    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.                   *
 *                                                                                  *
 ************************************************************************************/

#ifndef __database_sslrpc_driver_hpp
#define __database_sslrpc_driver_hpp

#include "database.hpp"
#include "databasedriver.hpp"
#include "dbresult.hpp"

#include <string>

#include <mynahsa/sslrpcclient.hpp>

/** Database driver interface for SSLRPC based connections.  This class provides a wrapper for
 *  all class Database methods.  It converts driver requests into networked objects that are transmitted
 *  using MynahSA to a server, which performs the real query on the single database instance for us.
 *  The results returned from the server to this class are de-serialized and returned.
 */
class DatabaseSSLRPCDriver : public DatabaseDriver { 
public:

  /**
   * Constructor - requires an active SSLRPCClient connection
   * @param client - the initialized connection
   * @return Nothing
   */
  DatabaseSSLRPCDriver(MynahSA::SSLRPCClient& client);
  
  
  /**
   * Destructor - closes connection
   * @return Nothing
   */
  ~DatabaseSSLRPCDriver();
  
  //! name search
  DBResult queryName(const std::string& nameStart, const std::string& nameEnd) const;
  
  //! age search
  DBResult queryAge(int ageStart, int ageEnd) const;
  
  //! eye color search
  DBResult queryEyeColor(PersonRecord::EyeColor) const;
  
  //! return all hits
  DBResult queryAll() const;
  
  //! modify a record, specified by uid
  DBResult modifyRecord(unsigned int uid, 
                        const std::string& name,
                        int age,
                        PersonRecord::EyeColor ec);
                        
  //! delete a record by uid                        
  DBResult deleteRecord(unsigned int uid);
  
  //! insert a record
  DBResult insertRecord(const std::string& name,
                        int age,
                        PersonRecord::EyeColor ec);

private:
  /**
   * Perform RPC and convert result from an IoBase to a DBResult, or throw a logic error
   * @param req - input request
   * @return DBResult 
   */
  DBResult rpcIt(const SHARED_PTR<MynahSA::IoBase>& req) const;
  
  //! sslrpc client
  mutable MynahSA::SSLRPCClient& _client;


};


#endif
