/************************************************************************************
 *    This file is part of the MynahSA streaming and archiving toolkit              *
 *    Copyright (C) 2006 Mynah-Software Ltd. All Rights Reserved.                   *
 *                                                                                  *
 *    This program is free software; you can redistribute it and/or modify          *
 *    it under the terms of the GNU General Public License, version 2               *
 *    as published by the Free Software Foundation.                                 *
 *                                                                                  *
 *    This program is distributed in the hope that it will be useful,               *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of                *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                 *
 *    GNU General Public License for more details.                                  *
 *                                                                                  *
 *    You should have received a copy of the GNU General Public License along       *
 *    with this program; if not, write to the Free Software Foundation, Inc.,       *
 *    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.                   *
 *                                                                                  *
 ************************************************************************************/

#ifndef __db_request_hpp
#define __db_request_hpp

#include <mynahsa/iobase.hpp>
#include <mynahsa/archive.hpp>

#include "personrecord.hpp"


/** Class DBRequest is an uber-request class.  We use it to represent all types of query request on 
 *  class Database so that we need not write a single class per request type on the database.  
 *  This class serves as a container for transporting both method request and data from client 
 *  applications to the database server.<br>
 *  <br>
 *  In general I do not advocate this style of design - Not all members are required for each query.
 *  I would recommend designing 7 classes (QueryAgeRequest, QueryNameRequest ...) in place where each
 *  datamember serves a specific purpose.<br>
 *  <br>
 *  This design was choosen for simplicity.
 */
class DBRequest : public MynahSA::IoBase {
public:
  //! the type of operation requested of the database
  enum RequestType { UNKNOWN,
                     QUERY_NAME, 
                     QUERY_AGE, 
                     QUERY_EYE_COLOR, 
                     QUERY_ALL, 
                     MODIFY_RECORD, 
                     DELETE_RECORD, 
                     INSERT_RECORD };
  
  /**
   * Constructor - create with default parameters
   * @return 
   */
  DBRequest();
  
  /**
   * Constructor - specify a specific request
   * @param request 
   * @return 
   */
  DBRequest(RequestType request);

  
  /**
   * Destructor
   * @return 
   */
  ~DBRequest();
  
  /**
   * return the ioName for this class
   * @return - std::string "DBRequest"
   */
  std::string ioName() const;
 
 
  /**
   * Serializer
   * @param ar - archive reference 
   */
  void serialize(MynahSA::Archive& ar);
 
 
  //! Request for db server
  RequestType _request;
  
  // we leave these directly in the public section and treat this class much like a c struct
  
  
  //! primary name - start name (or low order name) for queryName, or name for insertRecord/modifyRecord
  std::string _name;
  //! secondary name - only used as high order name for queryName
  std::string _name2;
  //! primary age - low age for queryAge, or age for insertRecord/modifyRecord
  int _age;
  //! secondary age - used only as high valued age for queryAge
  int _age2;

  //! unique identifier for a record, for deleteRecord and modifyRecord only
  unsigned int _uid;

  //! eye color for queryEyeColor or for insertRecord/modifyRecord
  PersonRecord::EyeColor _eyeColor;

};



#endif
