/************************************************************************************
 *    This file is part of the MynahSA streaming and archiving toolkit              *
 *    Copyright (C) 2006 Mynah-Software Ltd. All Rights Reserved.                   *
 *                                                                                  *
 *    This program is free software; you can redistribute it and/or modify          *
 *    it under the terms of the GNU General Public License, version 2               *
 *    as published by the Free Software Foundation.                                 *
 *                                                                                  *
 *    This program is distributed in the hope that it will be useful,               *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of                *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                 *
 *    GNU General Public License for more details.                                  *
 *                                                                                  *
 *    You should have received a copy of the GNU General Public License along       *
 *    with this program; if not, write to the Free Software Foundation, Inc.,       *
 *    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.                   *
 *                                                                                  *
 ************************************************************************************/

#ifndef __db_result_hpp
#define __db_result_hpp

#include "dbtypedef.hpp"

#include <mynahsa/iobase.hpp>
#include <mynahsa/archive.hpp>

#include <iostream>

/** class DBResult is returned from the database when operations are requested.  This class
 *  contains both error messages and hit lists.
 */
class DBResult : public MynahSA::IoBase { 
public:
  //! result code from a database operation
  enum Result { Undefined, Ok, Error };
   
  /**
   * Default constructor - configures an Undefined result
   * @return - nothing
   */
  DBResult();
   
  /**
   * Constructor
   * @param res - the result of from the database operation.
   * @param what - string message indicating error if one occurred
   * @param hits - the list of hits returned by a query.
   * @return 
   */
  DBResult(Result res, const std::string& what, const DBHitList& hits);

  /**
   * Copy constructor
   * @param result - the instance of DBResult to copy
   * @return 
   */
  DBResult(const DBResult& result);  

  /**
   * Destructor
   * @return - Nothing
   */
  ~DBResult();


  // reimplementation from MynahSA::IoBase
  
  /**
   * Get the name associated with this class
   * @return - the ioName string
   */
  std::string ioName() const;
  
  /**
   * store the DBResult onto the archiver ar
   * @param ar - the archiver to take the data from this class
   */
  void serialize(MynahSA::Archive& ar);

  /**
   * Gain access to hit list returned from the database by this query
   * @return const DBHitList reference
   */
  const DBHitList& getHits() const;
  
  //! in the event of an error, what contains the error string
  const std::string& what() const;
  
  //! get the result
  bool isOk() const { return _result == Ok; }
  
private:
  //! result of Database Operation
  Result _result;

  //! string stored to hold error message
  std::string _whatString;

  //! storage of hits
  DBHitList _hits;
  
};


/**
 * Output a DBResult class onto a std::ostream
 * @param os - output stream
 * @param r - the DBResult to output
 * @return - the output stream after placing DBResult onto it.
 */
std::ostream& operator<<(std::ostream& os, const DBResult& r);


#endif
