/************************************************************************************
 *    This file is part of the MynahSA streaming and archiving toolkit              *
 *    Copyright (C) 2006 Mynah-Software Ltd. All Rights Reserved.                   *
 *                                                                                  *
 *    This program is free software; you can redistribute it and/or modify          *
 *    it under the terms of the GNU General Public License, version 2               *
 *    as published by the Free Software Foundation.                                 *
 *                                                                                  *
 *    This program is distributed in the hope that it will be useful,               *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of                *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                 *
 *    GNU General Public License for more details.                                  *
 *                                                                                  *
 *    You should have received a copy of the GNU General Public License along       *
 *    with this program; if not, write to the Free Software Foundation, Inc.,       *
 *    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.                   *
 *                                                                                  *
 ************************************************************************************/

#include <iostream>

#include "parser.hpp"

#include <string>



#include <mynahsa/stringcast.hpp>

using namespace std;
using namespace MynahSA;




struct simple_parser { 
  // match a quoted string on s : \"*\".  Put string w/o quotes into v, update len
  bool parseQuotedString(int& len, string s, string& v) { 
    if (s[len] != '"') { 
      return false;
    }

    int i = len+1;
    while (i< s.size() && s[i] != '"') { 
      ++i;
    }
    if (i == s.size()) { 
      return false;
    }

    v = string(s, len+1, (i-1)-len );
    len = i+1;
    return true;
  }

  // match an integer, place int on intva, update len
  bool parseInt(int& len, string s, int& intval) { 
    try { 
      string k;
      int i=len;
      bool firstchar = true;
      while (i < s.size() && ((firstchar == true && s[i] == '-') || s[i] >= '0' && s[i] <= '9') ) { 
        k += s[i];
        firstchar == false;
        ++i;
      }
      if (i==0) { 
        return false;
      }
      intval = StringCast<int>(k);
      len = i;
      return true;
    } catch (logic_error& l) { 
      return false;
    }
  }

  // match an unsigned integer, place int on intva, update len
  bool parseUInt(int& len, string s, unsigned int& intval) { 
    try { 
      string k;
      int i=len;
      bool firstchar = true;
      while (i < s.size() && (s[i] >= '0' && s[i] <= '9') ) { 
        k += s[i];
        firstchar == false;
        ++i;
      }
      if (i==0) { 
	     return false;
      }
      intval = StringCast<unsigned int>(k);
      len = i;
      return true;
    } catch (logic_error& l) { 
      return false;
    }
  }  


  // accept an arbitrary number of spaces around a comma
  bool parseComma(int& len, string s) { 
    int ls = len;
    while (s[ls] == ' ') { 
      ++ls;
    }
      
    if (s[ls] == ',') { 
      ls += 1;
      while (s[ls] == ' ') {
        ++ls;
      }
      len = ls;
      return true;
    } else { 
      return false;
    }
  }

  // match a string as provided in match.  Update len if match is found
  bool matchString(int& len, string s, string match) {
    if (s.find(match,len) == len) {
      len += match.size();
      return true;
    } else { 
      return false;
    }
  }

  // parse an eyecolor {Blue/Green/Brown/Red), place in ec and update len
  bool parseEyeColor(int& len, string s, PersonRecord::EyeColor& ec) { 
    if (matchString(len, s, "Blue")) { 
      ec = PersonRecord::Blue;
      return true;
    } else  if (matchString(len, s, "Green")) { 
      ec = PersonRecord::Green;
      return true;
    }else  if (matchString(len, s, "Brown")) { 
      ec = PersonRecord::Brown;
      return true;
    }else  if (matchString(len, s, "Red")) { 
      ec = PersonRecord::Red;
      return true;
    } else {
      return false;
    }
  }


  // match the function call queryAll, and perform database action
  bool matchQueryAll(string s) { 
    int len= 0;
    if (matchString(len,s,"queryAll()")) { 
      _result = _dbd.queryAll();
      return true;
    } else { 
      return false;
    }
  }

  // match the function call queryName, and perform database action
  bool matchQueryName(string s) { 
    int len =0;
    string name1, name2;
    if (matchString(len,s, "queryName(") &&
      parseQuotedString(len, s, name1) &&
      parseComma(len, s) &&
      parseQuotedString(len, s, name2) &&
      matchString(len,s,")")) { 
      _result = _dbd.queryName(name1, name2);
      return true;
    } else { 
      return false;
    }
  }
  
  // match the function call queryAge, and perform database action
  bool matchQueryAge(string s) { 
    int len =0;
    int age1, age2;
    if (matchString(len,s, "queryAge(") &&
      parseInt(len, s, age1) &&
      parseComma(len,s) &&
      parseInt(len, s, age2) &&
      matchString(len,s,")")) { 
      _result = _dbd.queryAge(age1,age2);
      return true;
    } else { 
      return false;
    }
  }  

  // match the function call queryEyeColor, and perform database action
  bool matchQueryEyeColor(string s) { 
    PersonRecord::EyeColor ec;
    int len = 0;
    if (matchString(len, s, "queryEyeColor(") &&
      parseEyeColor(len,s,ec) &&	
      matchString(len,s,")")) { 
      _result = _dbd.queryEyeColor(ec);
      return true;
    } else { 
      return false;
    }
  }

  // match the function call insertRecord, and perform database action
  bool matchInsertRecord(string s) { 
    PersonRecord::EyeColor ec;
    string name;
    int age;
    int len=0;
    if (matchString(len, s, "insertRecord(") &&
        parseQuotedString(len, s, name) &&
        parseComma(len,s) &&
        parseInt(len, s, age) &&
        parseComma(len,s) &&
        parseEyeColor(len, s, ec) &&
        matchString(len, s, ")")) {
      _result = _dbd.insertRecord(name, age, ec);
      return true;
    }
    return false;
  }
  
  // match the function call modifyRecord, and perform database action
  bool matchModifyRecord(string s) { 
    PersonRecord::EyeColor ec;
    string name;
    int age;
    unsigned int uid;
    int len=0;
    if (matchString(len, s, "modifyRecord(") &&
        parseUInt(len, s, uid) &&
        parseComma(len,s) &&
        parseQuotedString(len, s, name) &&
        parseComma(len,s) &&
        parseInt(len, s, age) &&
        parseComma(len,s) &&
        parseEyeColor(len, s, ec) &&
        matchString(len, s, ")")) {
      _result = _dbd.modifyRecord(uid, name, age, ec);
      return true;
    }
    return false;
  }

  // match the function call modifyRecord, and perform database action
  bool matchDeleteRecord(string s) { 
    unsigned int uid;
    int len=0;
    if (matchString(len, s, "deleteRecord(") &&
        parseUInt(len, s, uid) &&
        matchString(len, s, ")")) {
      _result = _dbd.deleteRecord(uid);
      return true;
    }
    return false;
  }

  // match help function call
  bool matchHelp(string s) { 
    int len = 0;
    if (matchString(len, s, "help()") ||
        matchString(len, s, "help")) { 
      cerr << "Database Command Parser, acceptable commands: " << endl;
      cerr << "insertRecord(<quoted string>,<int>,<{Red, Green, Blue, Brown}>) - insert a record" << endl;
      cerr << "deleteRecord(<unique record identifier>) - delete a record" << endl;
      cerr << "modifyRecord(<unique record identifier>,<quoted string>,<int>,<{Red, Green, Blue, Brown}>) - modify a record" << endl;
      cerr << "queryAll() - show all records" << endl;
      cerr << "queryName(<quoted string>,<quoted string>) - query by name range" << endl;
      cerr << "queryAge(<int>,<int>) - query by age range" << endl;
      cerr << "queryEyeColor(<{Red, Green, Blue, Brown}>) - query by eye color" << endl;
      cerr << "help() - display this message" << endl;
      
      _result = DBResult(DBResult::Ok, "", DBHitList());
      return true;
    } 
    return false;
  }


  // accept all possible commands
  bool parse(string s) { 
    return 
      matchQueryAll(s) ||
      matchQueryName(s) ||
      matchQueryAge(s) ||
      matchInsertRecord(s) ||
      matchModifyRecord(s) ||
      matchDeleteRecord(s) ||
      matchHelp(s);
  }
  
  simple_parser(DatabaseDriver& dbd) : _dbd(dbd) { 
  }
  DatabaseDriver& _dbd;
  DBResult _result;

};


  
  
Parser::Parser(DatabaseDriver& dbd) : _driver(dbd)  {
}

DBResult Parser::operator()(const string& s) { 
  simple_parser cp(_driver);
  if (!cp.parse(s)) { 
    throw logic_error( string("Parse error in parsing string: ")+s );
  }
  return cp._result;
}



