/************************************************************************************
 *    This file is part of the MynahSA streaming and archiving toolkit              *
 *    Copyright (C) 2006 Mynah-Software Ltd. All Rights Reserved.                   *
 *                                                                                  *
 *    This program is free software; you can redistribute it and/or modify          *
 *    it under the terms of the GNU General Public License, version 2               *
 *    as published by the Free Software Foundation.                                 *
 *                                                                                  *
 *    This program is distributed in the hope that it will be useful,               *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of                *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                 *
 *    GNU General Public License for more details.                                  *
 *                                                                                  *
 *    You should have received a copy of the GNU General Public License along       *
 *    with this program; if not, write to the Free Software Foundation, Inc.,       *
 *    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.                   *
 *                                                                                  *
 ************************************************************************************/

#include <iostream>
#include <string>
#include "databasesslrpcdriver.hpp"
#include "parser.hpp"

#include "dbregister.hpp"

#include <mynahsa/stringcast.hpp>

#include <mynahsa/mynahsa.hpp>

using namespace std;
using namespace MynahSA;

void usage() { 
  
  cerr << "Usage: ssldbclient [-cert <certificate file> -pkey <private key file>] <machine name> <port number>" << endl;
  cerr << "       <ip address/ fully qualified machine name> is the target machine's name, running server." << endl;
  cerr << "       <port number> is the port number that the server is operating on." << endl;
  cerr << "       -cert <certficiate file> specify a certificate file" << endl;
  cerr << "       -pkey <private key file> specify the private key file associated with the certificate" << endl;
  
}

void parseClientArgs(int argc, char** argv,
                     string& machineName,
                     int& port,
                     string& certificateFile,
                     string& pkeyFile) { 
  bool gotHost = false;
  bool gotPort = false;
  
  int i = 1;
  while (i<argc) { 
    if (string(argv[i]) == "-cert") { 
      ++i;
      if (i>=argc) { 
        usage();
        exit(1);
      }
      certificateFile = argv[i];
    } else if (string(argv[i]) == "-pkey") { 
      ++i;
      if (i>=argc) { 
        usage();
        exit(1);
      }
      pkeyFile = argv[i];
    } else {
      if (!gotHost) { 
        gotHost = true;
        machineName = argv[i];
      } else if (!gotPort) { 
        gotPort = true;
        port = StringCast<int>(argv[i]);
      } else { 
        usage();
        exit(1);
      }
    }
    ++i;
  }
  if (!gotPort || !gotHost) {  
    usage();
    exit(1);
  }
}                     


int main(int argc, char** argv) {
  // initialize ssl 
  saInit();
  
  try { 
  
    string certificateFile = "";
    string pkeyFile = "";
    string machineName ="";
    int port=0;
    
    // parse arguments
    parseClientArgs(argc, argv, machineName, port, certificateFile, pkeyFile);
  
    // create the constructor object.
    DBRegister myConstructor;
    
    cerr << "Performing setup" << endl;

    // construct ssl rpc client
    SSLRPCClient myClient(myConstructor, machineName, port, certificateFile, pkeyFile);
  
    DatabaseSSLRPCDriver myDriver(myClient);
    
    Parser myParser(myDriver);
    
    string cmd;
    cout << "> ";
    cout.flush();
    while (getline(cin, cmd)) {
      try {
        if (!cmd.empty()) { 
          DBResult res = myParser(cmd);
          cout << res << endl;
        }
      } catch (logic_error& le) { 
        cerr << le.what() << endl;
      }
      cout << "> ";
      cout.flush();
    }
  } catch(StringCastException& bc) { 
    cerr << "Port number could not be converted into an integer.  Check arguments!" << endl;
    return 1;
  } catch (ExceptionBase& eb) { 
    cerr << "Caught a fatal exception: " << eb.what() << endl;
    return 1;
  } catch (...) { 
    cerr << "Uncaught exception.  Check and debug!" << endl;
    return 1;
  }
  return 0;
}
