/************************************************************************************
 *    This file is part of the MynahSA streaming and archiving toolkit              *
 *    Copyright (C) 2006 Mynah-Software Ltd. All Rights Reserved.                   *
 *                                                                                  *
 *    This program is free software; you can redistribute it and/or modify          *
 *    it under the terms of the GNU General Public License, version 2               *
 *    as published by the Free Software Foundation.                                 *
 *                                                                                  *
 *    This program is distributed in the hope that it will be useful,               *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of                *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                 *
 *    GNU General Public License for more details.                                  *
 *                                                                                  *
 *    You should have received a copy of the GNU General Public License along       *
 *    with this program; if not, write to the Free Software Foundation, Inc.,       *
 *    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.                   *
 *                                                                                  *
 ************************************************************************************/

#include <iostream>

#include "database.hpp"
#include "dbregister.hpp"
#include "dbrequestresponsehandler.hpp"

#include <signal.h>

#include <mynahsa/mynahsa.hpp>


using namespace std;

using namespace MynahSA;

void usage() {
  cerr << "Usage: " << endl;
  cerr << " server  [-port <int>] [-cert <certificate file> -pkey <private key file>] [-CAfile <CA file>]" << endl;
  cerr << " where:" << endl;
  cerr << "   -port <int> specifies the port to operate on" << endl;
  cerr << "   -cert <certificate file> specifies the certificate file" << endl;
  cerr << "   -pkey <private key file> specifies the private key file" << endl;
  cerr << "   -CAfile <CA file> specify a certificate authority file. See" << endl;
  cerr << "    man SSL_CTX_load_verify_locations for information on preparing the CA file." << endl;
  cerr << " Note: both -cert and -pkey must be specified, either one alone is an error" << endl;
  cerr << "       Omitting both -cert and -pkey will cause the server to create a certificate" << endl;
  cerr << "       on the fly with default parameters." << endl;        
  exit(1);
}


// this global is used to terminate the main loop once a ctrl-break is issued
bool running = true;

void signalHandler(int) {
  cerr << "Ctrl-break received, terminating!" << endl;
  running = false;
}


void parseArgs(int argc, char** argv, 
               int& serverPort, 
               string& certFile, 
               string& privKeyFile, 
               string& caFile) {
               
  bool privKeySpecified = false;
  bool certSpecified = false;
  int i=1;
  while (i<argc) { 
    if (string(argv[i]) == "-port") { 
      i++;
      if (i >= argc) { 
        usage();
      }
      serverPort = StringCast<int>(argv[i]);
    } else if (string(argv[i]) == "-cert") { 
      i++;
      if (i >= argc) { 
        usage();
      }
      certSpecified = true;
      certFile = argv[i];
    } else if (string(argv[i]) == "-pkey") { 
      i++;
      if (i >= argc) { 
        usage();
      }
      privKeySpecified = true;
      privKeyFile = argv[i];
    } else if (string(argv[i]) == "-CAfile") { 
      i++;
      if (i >= argc) { 
        usage();
      }
      caFile = argv[i];
    } else { 
      usage();
    }
    i++;
  }

  if ((privKeySpecified && !certSpecified) ||
      (!privKeySpecified && certSpecified)) { 
    usage();
  }
  // otherwise, all args parsed ok.
}

int main(int argc, char** argv) { 
  signal(SIGINT, signalHandler);

  // initialize archiving and streaming library
  saInit();

  X509* cert = 0;                          // cert pointer
  EVP_PKEY* pkey = 0;                      // pkey pointer
  try { 
    // hard assign the server port for this example
    int serverPort = 7501;
    string certFile = "automatically generated";
    string privKeyFile = "automatically generated";
    string caFile = "";

    parseArgs(argc, argv, 
              serverPort, 
              certFile, 
              privKeyFile, 
              caFile);
              
    // step 0: Initialize the database
    Database db("testdb.dat");

    // step 1: build the ping request response handler
    DBRequestResponseHandler drh(db);
    
    // step 2: create the stream registration object
    DBRegister myConstructor;
    
    if (certFile != "automatically generated") {
      // step 3: create a connection verification object
      // certificate standard identifier - forces client certificate CN to be identical to the 
      // name of the client which is obtained from name lookup!
      SSLConnectionCertVerifier myVerifier;
    
      // step 4: instantiate a connection manager on the handler and the stream constructor
      SSLConnectionManager cm(drh,
                              myConstructor,
                              myVerifier); 
                                                
      // step 5: build an ssl server on top of the connection manager instance
      SSLRPCServer server(cm, certFile, privKeyFile, serverPort,caFile);
      
      // step 6: enter loop where we serve requests forever
      while (running) { 
        try { 
          server.checkClients(5);
        } catch (const ServerConnectionException& sce) { 
          cerr << "Connection error detected: " << sce.what() << endl;
        }
      }
    } else { 
      // step 3: create a connection verification object
      // Default verifier - accepts everything
      SSLConnectionVerifier myVerifier;
      
      // step 4: instantiate a connection manager on the handler and the stream constructor
      SSLConnectionManager cm(drh,
                              myConstructor,
                              myVerifier); 
                                                
      // step 5: build an ssl server on top of the connection manager instance
      mkcert(&cert, &pkey, 1024, 0, 760);      // create cert and pkey - completely self signed 1024 bit
      SSLRPCServer server(cm, cert, pkey, serverPort,caFile); 
      
      // step 6: enter loop where we serve requests forever
      while (running) { 
        try { 
          server.checkClients(5);
        } catch (const ServerConnectionException& sce) { 
          cerr << "Connection error detected: " << sce.what() << endl;
        }
      }      
    }
  } catch (const ServerException& se) { 
    cerr << "Fatal server error occurred: " << se.what() << endl;
  }
  // cleanup on error
  if (pkey != 0) {
    EVP_PKEY_free(pkey);
  }
  
  if (cert != 0) { 
    X509_free(cert);
  }
}
