/************************************************************************************
 *    This file is part of the MynahSA streaming and archiving toolkit              *
 *    Copyright (C) 2006 Mynah-Software Ltd. All Rights Reserved.                   *
 *                                                                                  *
 *    This program is free software; you can redistribute it and/or modify          *
 *    it under the terms of the GNU General Public License, version 2               *
 *    as published by the Free Software Foundation.                                 *
 *                                                                                  *
 *    This program is distributed in the hope that it will be useful,               *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of                *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                 *
 *    GNU General Public License for more details.                                  *
 *                                                                                  *
 *    You should have received a copy of the GNU General Public License along       *
 *    with this program; if not, write to the Free Software Foundation, Inc.,       *
 *    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.                   *
 *                                                                                  *
 ************************************************************************************/

#ifndef __archive_hpp
#define __archive_hpp

#include <mynahsa/spimpl.hpp>

#include <string>
#include <vector>
#include <list>
#include <utility>
#include <map>
#include <set>

#include <mynahsa/exceptionbase.hpp>

#include <mynahsa/type_traits.hpp>

namespace MynahSA { 
  class IoBase;


  /** Archive is a pure base for classes IArchive and OArchive for input 
  *  and output respectively.
  */
  class Archive { 
  public:
  
   /** class ArchiveConstructor is a base for constructor functions
    *  responsible for instantiating objects off of a stream when a base
    *  pointer is provided through shared pointer.
    */
    class ArchiveConstructor { 
    public:
      //! constructor functor constructor
      ArchiveConstructor() {
      }
  
      //! destructor
      ~ArchiveConstructor() {
      }
  
      //! operator() is used to construct the object - pure base
      virtual SHARED_PTR<IoBase> operator()() const = 0;
    };
 
  
    //! constructor
    Archive();
    
    //! copy constructor - copy constructor functions
    Archive(const Archive& ar);
    
    //! construct with a pre-existing map of constructor functions
    Archive(const std::map<std::string, SHARED_PTR<Archive::ArchiveConstructor> >& cons);
  
    //! virtual destructor
    virtual ~Archive();
  
  
   /** register a constructor object with this iarchive instance - this is necessary
    *  for restoration of objects pointed to by shared pointers.
    *
    *  Note: Ownership of the IArchiveConstructor is taken by this class.
    */
    void registerConstructor(const std::string& name, const SHARED_PTR<ArchiveConstructor>& cons) {
      _constructors[name] = cons;
    }
 
    //! provide external access to constructor functions
    const std::map<std::string, SHARED_PTR<ArchiveConstructor> >& getConstructors() const { 
      return _constructors; 
    }
    
    // must treat int / bool / float / double / char
  
    //! archive long long
    virtual Archive& operator&(long long&) = 0;
    
    //! archive unsigned long long
    virtual Archive& operator&(unsigned long long&) = 0;
  
    //! archive unsigned int
    virtual Archive& operator&(unsigned int&) = 0;
  
    //! archive int
    virtual Archive& operator&(int&) = 0;
    
    //! archive short
    virtual Archive& operator&(short&) = 0;
    
    //! archvie ushort
    virtual Archive& operator&(unsigned short&) = 0;
    
    //! archive char
    virtual Archive& operator&(char&) = 0;
    
    //! archive unsigned char
    virtual Archive& operator&(unsigned char&) = 0;
    
    //! archive bool
    virtual Archive& operator&(bool&) = 0;
    
    //! archive float
    virtual Archive& operator&(float&) = 0;
    
    //! archive double
    virtual Archive& operator&(double&) = 0;
  
    //! archive std::string
    virtual Archive& operator&(std::string&) = 0;
  
    //! archive a std::vector
    template<class T>
    Archive& operator&(std::vector<T>& v) { 
      // these statements work around the issues that this method is used for both read and write
      // step 1: copy the size
      int size = v.size();
      // step 2: archive the size - NOTE:
      //        In the IArchive case - size will be overwritten
      //        In the OArchive case - size will be v.size()
      (*this) & size;
      //        In IArchive case, resize the vector appropriately
      if (size > v.size()) {
        v.resize(size);
      }
      // now recover the data
      for (unsigned int i=0; i<v.size(); i++) { 
        (*this) & v[i];
      }
      return *this;
    }
  
    //! archive a std::list
    template<class T>
    Archive& operator&(std::list<T>& l) { 
      // This method contains more trickery like the vector storage routine above.  The key difference
      // here is that the symmetric read/write operator used for the vector (operator[]) is unavailable
      // for lists - even if it were available, the cost would be order N in the list length each time
      // a list item were accessed.  To circumvent this problem we change the loop control to use
      // iterators as each item has a symmetric operator defined on it - Archive's operator&
      int size = l.size();
      (*this) & size;
      if (size > l.size()) { 
        l.resize(size);
      }
      for (typename std::list<T>::iterator lit = l.begin();
          lit != l.end();
          ++lit) { 
        (*this) & (*lit);
      }
      return *this;
    }
    
    //! Archive a std::pair
    template<class a_type, class b_type>
    Archive& operator&(std::pair<a_type, b_type>& p) { 
      (*this) & p.first;
      (*this) & p.second;
      return *this;
    }
    
    
    //! archive a std::map - store indicies and data
    template<class value_type>
    Archive& operator&(std::set<value_type>& s) { 
      // Without any symmetric read/write operator we are forced to split the archive method into two
      // paths for std::set.  We rely on a virtual member getArchiveMode() to inform us of the data direction.
      // If we are reading, we use the read path, otherwise, the write path.
      //
      // When reading, we recover the number of elements first, then for each element, inarchive a 
      // value_type and insert that entry into the set.
      // 
      // When writing, we simply store the size, and loop over the entire map, writing each 
      // value_type onto the archive
      if (getArchiveMode() == ARCHIVE_READ) { 
        // step 1: clear the current map
        s.clear();
        // step 2: recover the size off the stream
        int size = 0;
        (*this) & size; // get the amount of data off the stream
        // step 3: for each element on the stream -
        for (unsigned int i=0; i < size; i++) { 
          // step 3a: recover an element
          value_type element;
          (*this) & element;
          // step 3b: store element into map.
          s.insert(element);
        }
      } else { 
        // step 1: archive the size
        int size = s.size();
        (*this) & size;  
        // step 2: archive each element
        for (typename std::set<value_type>::const_iterator sit = s.begin();
            sit != s.end();
            ++sit) { 
          value_type element = (*sit);  // unfortunate copy
          (*this) & element;  // use std::pair archiver to store element
        }
      }
      return *this;
    }
    
    
    //! archive a std::map - store indicies and data
    template<class index_type, class value_type>
    Archive& operator&(std::map<index_type, value_type>& m) { 
      // Without any symmetric read/write operator we are forced to split the archive method into two
      // paths for std::map.  We rely on a virtual member getArchiveMode() to inform us of the data direction.
      // If we are reading, we use the read path, otherwise, the write path.
      //
      // When reading, we recover the number of elements first, then for each element, inarchive a 
      // pair of index_type, value_type and insert that entry into the map.
      // 
      // When writing, we simply store the size, and loop over the entire map, writing each 
      // std::pair<index_type, value_type> onto the archive
      if (getArchiveMode() == ARCHIVE_READ) { 
        // step 1: clear the current map
        m.clear();
        // step 2: recover the size off the stream
        int size = 0;
        (*this) & size; // get the amount of data off the stream
        // step 3: for each element on the stream -
        for (unsigned int i=0; i < size; i++) { 
          // step 3a: recover an element
          std::pair<index_type, value_type> element;
          (*this) & element;
          // step 3b: store element into map.
          m.insert(element);
        }
      } else { 
        // step 1: archive the size
        int size = m.size();
        (*this) & size;  
        // step 2: archive each element
        for (typename std::map<index_type, value_type>::const_iterator mit = m.begin();
            mit != m.end();
            ++mit) { 
          // this copy is unfortunate - it seems that std::map reinterprets std::string as std::basic_string<char, ...> which does not match our stream operator for std::string when compiling with GCC
    //BMS20060611 - fix for Solaris compiler, copy both parameters.
    //  NOTE: the desired effect here may be achieveable with a
    //        reinterpret_cast and/or const cast, and could avoid copying.
          std::pair<index_type, value_type> element( (*mit).first, (*mit).second );
          (*this) & element;  // use std::pair archiver to store element
        }
      }
      return *this;
    }
  
  //! archive a std::map - store indicies and data
    template<class index_type, class value_type>
    Archive& operator&(std::multimap<index_type, value_type>& m) { 
      // Without any symmetric read/write operator we are forced to split the archive method into two
      // paths for std::map.  We rely on a virtual member getArchiveMode() to inform us of the data direction.
      // If we are reading, we use the read path, otherwise, the write path.
      //
      // When reading, we recover the number of elements first, then for each element, inarchive a 
      // pair of index_type, value_type and insert that entry into the map.
      // 
      // When writing, we simply store the size, and loop over the entire map, writing each 
      // std::pair<index_type, value_type> onto the archive
      if (getArchiveMode() == ARCHIVE_READ) { 
        // step 1: clear the current map
        m.clear();
        // step 2: recover the size off the stream
        int size = 0;
        (*this) & size; // get the amount of data off the stream
        // step 3: for each element on the stream -
        for (unsigned int i=0; i < size; i++) { 
          // step 3a: recover an element
          std::pair<index_type, value_type> element;
          (*this) & element;
          // step 3b: store element into map.
	  std::pair<const index_type, value_type> cElement(element.first, element.second);
          m.insert(cElement);                               // NOTE: implicit use of copy constructor
        }
      } else { 
        // step 1: archive the size
        int size = m.size();
        (*this) & size;  
        // step 2: archive each element
        for (typename std::multimap<index_type, value_type>::const_iterator mit = m.begin();
            mit != m.end();
            ++mit) { 
          // this copy is unfortunate - it seems that std::map reinterprets std::string as std::basic_string<char, ...> which does not match our stream operator for std::string when compiling with GCC
          //BMS20060611 - fix for Solaris compiler, copy both parameters.
          //  NOTE: the desired effect here may be achieveable with a
          //        reinterpret_cast and/or const cast, and could avoid copying.
          std::pair<index_type, value_type> element( (*mit).first, (*mit).second );
          (*this) & element;  // use std::pair archiver to store element
        }
      }
      return *this;
    }
  
  


    /** Archive operator& for shared pointer types.  Note that this code path is forked depending on whether
     *  or not the type being serialized is a fundamental type (fundamental_t) e.g. int, char, etc... or
     *  a complex type.  Complex types must derive from class IoBase;
     */
    template<class T>
    Archive& operator&(SHARED_PTR<T>& ptr) { 
      return archiveSP(ptr, fundamental_t<T>());
    }


    /** A template class handler - objects must be either enum type (as identified by is_enum , or
     *  provide a serialize method
     *
     * \note archiving of array types is supported on gcc, and Microsoft VC++ 2003+ using the following
     *       template:
     *
     * \code
     *  ar & arrayType;   // where ar is an archive instance
     * \endcode
     *
     *       on SUNpro 5.8 the compiler cannot differentiate the use of operator& and the "address of"
     *       operator.  Therefore, the following code fragment must be used:
     * \code
     *  ar.operator&(arrayType);
     * \endcode
     */
    template<class T>
    Archive& operator&(T& x) {
      // a bit of template meta-programming, 
      serializeArray(x, MynahSA::array_t<T>());
      return *this;
    }
    
    

    
    
    /** clearUPR - remove all unique pointer references from the archive, and free any associated
     *  memory by removing dangling SHARED_PTRs.  This method servers an important purpose: When 
     *  objects are serialized by SHARED_PTR, Archive holds a copy of the shared pointer.  This is
     *  necessary s.t. multiple pointers to the same object do not cause multiple object to be 
     *  transmitted.  clearUPR must be called to free all such references.
     *
     *  To make this explicit, here are two code sequences with explaination of what is being
     *  transmitted:
     *  <pre>
     *    Archive& archive;  // some non-pure derivitive of MynahSA::Archive
     *    SHARED_PTR<ObjectType> sp1; // a shared pointer to a real instance
     *    archive << sp1;                 // transmit sp1
     *    archive << sp1;                 // transmit sp1
     *  </pre>
     *  This sequence causes the following information to be transmitted on the stream:
     *  [shared_pointer to ObjectType [(*sp1) details]] [unique pointer reference to sp1]
     *  whereas, this code sequence:
     *  <pre>
     *    Archive& archive;  // some non-pure derivitive of MynahSA::Archive
     *    SHARED_PTR<ObjectType> sp1; // a shared pointer to a real instance
     *    archive << sp1;                 // transmit sp1
     *    archive.clearUPR();             // remove all unique pointer references
     *    archive << sp1;                 // transmit sp1
     *  </pre>
     *  caues the following to be emitted on the stream:
     *  [shared_pointer to ObjectType [(*sp1) details]] [shared_pointer to ObjectType [(*sp1) details]]
     */
    void clearUPR();
    
    /** touch a pointer - this indicates to the archiver that it should place a real pointer onto 
     * the stream and not a pointer reference.
     */
    template<class T>
    void touchPtr(const SHARED_PTR<T>& ptr) { 
      SHARED_PTR<void> value(STATIC_PTR_CAST<void>(ptr)); // fetch pointer value
      std::map<SHARED_PTR<void>, unsigned int>::iterator mit = _ptrToUPR.find(value);  // find its universal pointer value
      if (mit != _ptrToUPR.end()) { 
        unsigned int upr = (*mit).second;
        _ptrToUPR.erase(mit);  // get rid of the upr and pointer associated
        std::map<unsigned int, SHARED_PTR<void> >::iterator mit2 = _uprToPtr.find(upr);
        if (mit2 != _uprToPtr.end()) {
          _uprToPtr.erase(mit2);
        }
      }
    }
    
  protected:
    enum ArchiveMode { ARCHIVE_READ, ARCHIVE_WRITE };
    //! return the archive mode - reader or writer.  This is necessary for archiving certain STL
    //! types
    virtual ArchiveMode getArchiveMode() const =0;
    
    
  private:
  
    //! serialization of non-array types; just pass to serialize term
    template<class T>
    void serializeArray(T& x, const false_t&) {
      // a bit of template meta-programming, 
      serializeTerm(x, MynahSA::enum_t<T>());
    }
    
    //! serialization of array types; just serialize each element
    template<class T>
    void serializeArray(T& x, const true_t&) { 
      for (unsigned int i=0; i<sizeof(T); i++) { 
        (*this) & x[i];
      }
    }  
 
  
    //! serialization of enum types - invoked from operator&(T&);
    template<class T> 
    void serializeTerm(T& x, const true_t&) { 
      (*this) & ((int&) x);
    }
    
    //! serialization of non-enum types - invoked from operator&(T&);
    template<class T>
    void serializeTerm(T& x, const false_t&) { 
      x.serialize(*this);
    }
  
  
    //! Archive operator for shared pointers to fundamentals
    template<class T>
    Archive& archiveSP(SHARED_PTR<T>& instance, const true_t&) {
      
      if (getArchiveMode() == ARCHIVE_READ) { 
        // read path
        
        // step 1: Determine if the sender is sending a UPR or a real object 
        bool useUPR;
        (*this) & useUPR;
        if (useUPR) { 
          // fetch the UPR
          unsigned int upr;
          (*this) & upr;
          std::map< unsigned int, SHARED_PTR<void> >::const_iterator mit = _uprToPtr.find(upr);
          
          if (mit == _uprToPtr.end()) {
            throw ExceptionBase("Archive::operator&(SHARED_PTR<T>&) - recovery error: Unique Pointer Reference does not exist.");
          }
          
          // ok, UPR was found - perform shared pointer cast - return pointer
          instance = STATIC_PTR_CAST<T>( (*mit).second );
        } else { 
          // We're not receiving a UPR; we must recover an entire object
          
          // constructor function performs archive recovery operation
          instance = SHARED_PTR<T>(new T);
          
          // now - we must store the instance; as we may require this again later, by upr
          // we do not have a complex protocol for agreeing on UPRs between transmitter and receiver, rather,
          // the algorithm is sequential increment - therefore, everytime we transmit an object, we increment the
          // transmitter's UPR counter, and everytime we receive a new object we increment the UPR count...
          // Both algorithms play out in sequence on the transmitter and receiver so there are no problems
          // with UPRs getting out of sequence.
          
          SHARED_PTR<void> internalPointer = STATIC_PTR_CAST<void>(instance);  // convert to a void ptr
          // store on pointer to upr map
          _ptrToUPR[internalPointer] = _nextUPR;
          // store on upr to pointer map
          _uprToPtr[_nextUPR] = internalPointer;
          // increment upr count
          ++_nextUPR;
          
          // recover the object from the stream
          (*this) & (*instance);  // restore object pointed to by fundamental type pointer
        }
      } else { 
        // write path
        
        // step 1: determine if we've previously transmitted this object
        SHARED_PTR<void> internalPointer = STATIC_PTR_CAST<void>(instance); // convert to void ptr
        
        // step 2: find pointer on ptrToUPR map
        std::map< SHARED_PTR<void>, unsigned int >::const_iterator mit = _ptrToUPR.find(internalPointer);
        if (mit != _ptrToUPR.end()) { 
          // transmit the UPR and do not transmit the object again
          bool trueBool = true;
          (*this) & trueBool;  // transmit true to indicate we are transmitting a UPR
          unsigned int upr = (*mit).second;
          (*this) & upr;       // transmit the unique pointer reference
        } else { 
          bool falseBool = false;
          (*this) & falseBool;  // we're not sending a UPR, but the whole object
        
          // note: with fundamental_t(s) we don't send a string for the object name

          // now - store the UPR for this object.  We must do this prior to transmitting the object
          // if we did not do this in order, we could endup in a recursive loop.
          //
          // Another way of thinking about this step is that it breaks the graph structure, so back-edges
          // pointing at this node are now effectively broken; we won't re-traverse this node.
          _ptrToUPR[internalPointer] = _nextUPR;
          _uprToPtr[_nextUPR] = internalPointer;
          ++_nextUPR;
          
          // and transmit the instance
          (*this) & (*instance);
        }
      }
      return *this;
    }
  
  
  
  
    //! Generic archive operator - for transmitting descendants of IoBase
    template<class T>
    Archive& archiveSP(SHARED_PTR<T>& instance, const false_t&) {
#ifdef DEBUG
      // this is basically an assertion - that instance is a descendant of IoBase
      STATIC_PTR_CAST<IoBase>(instance);
#endif      
      
      if (getArchiveMode() == ARCHIVE_READ) { 
        // read path
        
        // step 1: Determine if the sender is sending a UPR or a real object 
        bool useUPR = false;
        (*this) & useUPR;
        if (useUPR) { 
          // fetch the UPR
          unsigned int upr;
          (*this) & upr;
          std::map< unsigned int, SHARED_PTR<void> >::const_iterator mit = _uprToPtr.find(upr);
          
          if (mit == _uprToPtr.end()) {
            throw ExceptionBase("Archive::operator&(SHARED_PTR<T>&) - recovery error: Unique Pointer Reference does not exist.");
          }
          
          // ok, UPR was found - perform shared pointer cast - return pointer
          instance = STATIC_PTR_CAST<T>( (*mit).second );
        } else { 
          // We're not receiving a UPR; we must recover an entire object
          std::string name; // get the class name
          (*this) & name;  
          SHARED_PTR<ArchiveConstructor> iac = _constructors[name];
  
          if (!iac) {
            throw ExceptionBase(std::string("Class name: ")+name+" is not registered");
          }
          // constructor function performs archive recovery operation
          instance = STATIC_PTR_CAST<T>( (*iac)() );
          
          // now - we must store the instance; as we may require this again later, by upr
          // we do not have a complex protocol for agreeing on UPRs between transmitter and receiver, rather,
          // the algorithm is sequential increment - therefore, everytime we transmit an object, we increment the
          // transmitter's UPR counter, and everytime we receive a new object we increment the UPR count...
          // Both algorithms play out in sequence on the transmitter and receiver so there are no problems
          // with UPRs getting out of sequence.
          
          SHARED_PTR<void> internalPointer = STATIC_PTR_CAST<void>(instance);  // convert to a void ptr
          // store on pointer to upr map
          _ptrToUPR[internalPointer] = _nextUPR;
          // store on upr to pointer map
          _uprToPtr[_nextUPR] = internalPointer;
          // increment upr count
          ++_nextUPR;
          
          // recover the object from the stream
          instance->serialize(*this);
                    
        }
      } else { 
        // write path
        
        // step 1: determine if we've previously transmitted this object
        SHARED_PTR<void> internalPointer = STATIC_PTR_CAST<void>(instance); // convert to void ptr
        
        // step 2: find pointer on ptrToUPR map
        std::map< SHARED_PTR<void>, unsigned int >::const_iterator mit = _ptrToUPR.find(internalPointer);
        if (mit != _ptrToUPR.end()) { 
          // transmit the UPR and do not transmit the object again
          bool trueBool = true;
          (*this) & trueBool;  // transmit true to indicate we are transmitting a UPR
          unsigned int upr = (*mit).second;
          (*this) & upr;       // transmit the unique pointer reference
        } else { 
          bool falseBool = false;
          (*this) & falseBool;  // we're not sending a UPR, but the whole object
        
          // prevent problems when converting value into reference
          std::string copyRef = instance->ioName();         // transmit object name
          (*this) & copyRef; // store the io name

          
          // now - store the UPR for this object.  We must do this prior to transmitting the object
          // if we did not do this in order, we could endup in a recursive loop.
          //
          // Another way of thinking about this step is that it breaks the graph structure, so back-edges
          // pointing at this node are now effectively broken; we won't re-traverse this node.
          _ptrToUPR[internalPointer] = _nextUPR;
          _uprToPtr[_nextUPR] = internalPointer;
          ++_nextUPR;
          
          // and transmit the instance
          instance->serialize( (*this) );
          
        }
      }
      return *this;
    }
  
  
  
  
  
  
  
  
  
  
  
    //! constructors contains a map of object ids (ioNames) to constructor functions
    std::map<std::string, SHARED_PTR<ArchiveConstructor> > _constructors;
    
    //! pointer to universal pointer reference map
    std::map< SHARED_PTR<void>, unsigned int > _ptrToUPR;
    
    //! universal pointer reference to pointer map
    std::map< unsigned int, SHARED_PTR<void> > _uprToPtr;
    
    //! next UPR to issue in
    unsigned int _nextUPR;
  
  };
};



#endif
