/************************************************************************************
 *    This file is part of the MynahSA streaming and archiving toolkit              *
 *    Copyright (C) 2006 Mynah-Software Ltd. All Rights Reserved.                   *
 *                                                                                  *
 *    This program is free software; you can redistribute it and/or modify          *
 *    it under the terms of the GNU General Public License, version 2               *
 *    as published by the Free Software Foundation.                                 *
 *                                                                                  *
 *    This program is distributed in the hope that it will be useful,               *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of                *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                 *
 *    GNU General Public License for more details.                                  *
 *                                                                                  *
 *    You should have received a copy of the GNU General Public License along       *
 *    with this program; if not, write to the Free Software Foundation, Inc.,       *
 *    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.                   *
 *                                                                                  *
 ************************************************************************************/

#ifndef __i_stream_base_hpp
#define __i_stream_base_hpp

#include <string>
#include <map>
#include <list>
#include <vector>

#include <stdexcept>

#include <mynahsa/exceptionbase.hpp>

namespace MynahSA { 
  /** Exception handling class for input SSL Stream errors.
  */
  class IStreamReadError : public ExceptionBase { 
  public:
    /**
     * Constructor 
     * @param c - error message
     * @return Nothing
     */
    IStreamReadError(const std::string& c);
    /**
     * Destructor
     * @return Nothing
     */
    ~IStreamReadError() throw();
  };
  
  
  /** place holder class for zero length reads - this is an indication that the stream
  *  read has stopped; client has closed the stream.  ISSLStreamReadNothing is not used
  *  to represent an error, but a normal exception: the stream has closed.
  */
  class IStreamEOF {
  public:
    //! constructor
    IStreamEOF();
  
    //! destructor
    ~IStreamEOF();
  };
  
  
  
  /** Class IStreamBase serves as a base class for all input streams.
  *  The class provides a primitive level interface for all c++ builtin
  *  types as pure virtual methods.  The base class also provides methods
  *  for streaming STL types using the virtual methods.
  */
  class IStreamBase { 
  public:
  
    //! constructor
    IStreamBase();
    
    //! virtual destructor
    virtual ~IStreamBase();
    
    //! instream a char
    virtual IStreamBase& operator>>(char&) = 0;
    
    //! instream an unsigned char
    virtual IStreamBase& operator>>(unsigned char&) = 0;
    
    //! instream a short
    virtual IStreamBase& operator>>(short&) = 0;
    
    //! instream an unsigned short
    virtual IStreamBase& operator>>(unsigned short&) = 0;
    
    //! instream a bool
    virtual IStreamBase& operator>>(bool&) = 0;
    
    //! instream an int
    virtual IStreamBase& operator>>(int&) = 0;
    
    //! instream an unsigned int
    virtual IStreamBase& operator>>(unsigned int&) = 0;
    
    //! instream a long long
    virtual IStreamBase& operator>>(long long&) = 0;
    
    //! instream an unsigned long long
    virtual IStreamBase& operator>>(unsigned long long&) = 0;
    
    //! instream a double
    virtual IStreamBase& operator>>(double&) = 0;
    
    //! instream a float
    virtual IStreamBase& operator>>(float&) = 0;
    
    //! in streaming of std strings
    virtual IStreamBase& operator>>(std::string& s);
    
    
    //! get function implements character level input - it is unformatted - and will read 1 chr at a time
    virtual void get(char& c) = 0;
  
  };
};




#endif
