/************************************************************************************
 *    This file is part of the MynahSA streaming and archiving toolkit              *
 *    Copyright (C) 2006 Mynah-Software Ltd. All Rights Reserved.                   *
 *                                                                                  *
 *    This program is free software; you can redistribute it and/or modify          *
 *    it under the terms of the GNU General Public License, version 2               *
 *    as published by the Free Software Foundation.                                 *
 *                                                                                  *
 *    This program is distributed in the hope that it will be useful,               *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of                *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                 *
 *    GNU General Public License for more details.                                  *
 *                                                                                  *
 *    You should have received a copy of the GNU General Public License along       *
 *    with this program; if not, write to the Free Software Foundation, Inc.,       *
 *    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.                   *
 *                                                                                  *
 ************************************************************************************/

#ifndef __o_archive_hpp
#define __o_archive_hpp

#include <mynahsa/spimpl.hpp>

#include <string>

#include "archive.hpp"
#include "iobase.hpp"

namespace MynahSA { 

  template<class T>
  class OArchive : public Archive { 
  public:
    //! constructor
    OArchive(T& s) : _stream(s) { 
    }
  
    OArchive(const OArchive<T>& i) : Archive(i), 
                                     _stream(i._stream) {
    }
  
    //! destructor
    ~OArchive() { 
    }
 
  
  
    // must treat int / bool / float / double / char
    
    //! oarchive long long
    virtual Archive& operator&(long long& j) { 
      _stream << j; 
      return *this;
    }
    
    //! oarchive unsigned long long
    virtual Archive& operator&(unsigned long long& j) { 
      _stream << j;
      return *this;
    }
  
  
    //! archive unsigned int
    virtual Archive& operator&(unsigned int& i) {
      _stream << i;
      return *this;
    }
  
    //! archive int
    virtual Archive& operator&(int& i) { 
      _stream << i;
      return *this;
    }
  
    //! oarchive short
    virtual Archive& operator&(short& s) { 
      _stream << s;
      return *this;
    }
    
    //! oarchive ushort
    virtual Archive& operator&(unsigned short& s) { 
      _stream << s;
      return *this;
    }
  
  
    //! archive bool
    virtual Archive& operator&(bool& b) {
      _stream << b;
      return *this;
    }
    
    //! archive float
    virtual Archive& operator&(float& f) { 
      _stream << f;
      return *this;
    }
    
    //! archive double
    virtual Archive& operator&(double& d) {
      _stream << d;
      return *this;
    }
  
    //! archive char
    virtual Archive& operator&(char& c) { 
      _stream << c;
      return *this;
    }
  
    //! archive unsigned char
    virtual Archive& operator&(unsigned char& c) { 
      _stream << c;
      return *this;
    }
  
  
    //! archive std::string
    virtual Archive& operator&(std::string& s) { 
      _stream << s;
      return *this;
    }
  
    template <class Type>
    Archive& operator<<(SHARED_PTR<Type>& instance) { 
      (*this) & instance;
      return *this;
    }
    
    template <class Type>
    Archive& operator<<(Type& instance) { 
      instance.serialize(*this);
      return *this;
    }
    
    void flush() { 
      _stream.flush();
    }
  
    //! Using declaration to allow base class method usage
    using Archive::operator&;
  
  
    // OArchive must provide STL wrappers for operator<< - we can simply use Archive::operator& to do the work, but we must provide the function
    // override s.t. the compiler will have a method for implementing the unidirectional stream operator
    template<class Type>
    Archive& operator<<(std::list<Type>& l) { 
      (*this) & l;
      return *this;
    }
    
    template<class Type>
    Archive& operator<<(std::vector<Type>& v) { 
      (*this) & v;
      return *this;
    }
    
    template<class A, class B>
    Archive& operator<<(std::pair<A,B>& p) { 
      (*this) & p;
      return *this;
    }
    
    template<class index_type, class value_type>
    Archive& operator<<(std::map<index_type, value_type>& m) { 
      (*this) & m;
      return *this;
    }
    
    template<class index_type, class value_type>
    Archive& operator<<(std::multimap<index_type, value_type>& m) { 
      (*this) & m;
      return *this;
    }  

    template<class value_type>
    Archive& operator<<(std::set<value_type>& s) { 
      (*this) & s;
      return *this;
    }

  protected:
    //! provide archive READ/WRITE mode information.
    ArchiveMode getArchiveMode() const { 
      return Archive::ARCHIVE_WRITE;
    }
  
  
  private:
    T& _stream;
  
  };
};

#endif
