/************************************************************************************
 *    This file is part of the MynahSA streaming and archiving toolkit              *
 *    Copyright (C) 2006 Mynah-Software Ltd. All Rights Reserved.                   *
 *                                                                                  *
 *    This program is free software; you can redistribute it and/or modify          *
 *    it under the terms of the GNU General Public License, version 2               *
 *    as published by the Free Software Foundation.                                 *
 *                                                                                  *
 *    This program is distributed in the hope that it will be useful,               *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of                *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                 *
 *    GNU General Public License for more details.                                  *
 *                                                                                  *
 *    You should have received a copy of the GNU General Public License along       *
 *    with this program; if not, write to the Free Software Foundation, Inc.,       *
 *    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.                   *
 *                                                                                  *
 ************************************************************************************/

#ifndef __o_blowfish_stream_hpp
#define __o_blowfish_stream_hpp

#include <mynahsa/ostreambase.hpp>

#include <openssl/blowfish.h>

#include <iostream>

namespace MynahSA { 
  class OBlowfishStream : public OStreamBase {
  public:
    
    /** Constructor - you specify the std::output stream and key.  Key strength depends on the 
     *                on the length of the key.  Refer to: http://www.openssl.org/docs/crypto/blowfish.html# 
     *                for further details.
     * 
     * @param os Output stream (an instance of std::ostream)
     * @param key Key for encrypting the data.
     * @return 
     */
    OBlowfishStream(std::ostream& os, const std::string& key);
    
    //! copy constructor
    OBlowfishStream(const OBlowfishStream& os);
    
    //! Destructor
    virtual ~OBlowfishStream();
    
    //! ostream char
    virtual OStreamBase& operator<<(const char);
    
    //! ostream unsigned char
    virtual OStreamBase& operator<<(const unsigned char);
    
    //! ostream short
    virtual OStreamBase& operator<<(const short);
    
    //! ostream unsigned short
    virtual OStreamBase& operator<<(const unsigned short);
    
    //! ostream bool
    virtual OStreamBase& operator<<(const bool);
    
    //! ostream int
    virtual OStreamBase& operator<<(const int);
    
    //! ostream uint
    virtual OStreamBase& operator<<(const unsigned int);
    
    //! ostream long long
    virtual OStreamBase& operator<<(const long long);
    
    //! ostream unsigned long
    virtual OStreamBase& operator<<(const unsigned long long);
    
    //! ostream double
    virtual OStreamBase& operator<<(const double);
    
    //! ostream float
    virtual OStreamBase& operator<<(const float);   
        
    // using declaration to make base class members accessible
    using OStreamBase::operator<<;

    //! method for closing stream
    void close();

    
  private:
    /** flush is a private member for this class - because encryption is done on 8 byte blocks
     *  and calling flush will cause a number of "zero" bytes to be inserted onto the end of the
     *  stream to pad the lenght out to 64 bits.  Therefore, arbitrary calling of flush will cause
     *  pad zeros to be inserted incorrectly onto the stream.  This would break input streaming, and
     *  therefore, this method is private.
     */
    void flush();
    
    //! write a single character into the buffer.
    void bfWriteChar(unsigned char value);

    //! Reference to provided std::ostream
    std::ostream& _os;
    
    //! local buffer - 64 bits for BF_cbc_encrypt
    unsigned char _buffer[8];
    
    //! output buffer
    unsigned char _outBuffer[8];
    
    //! buffer data for 
    unsigned char _ivec[8];
    
    //! buffer pointer - points to the current back of the buffer
    unsigned int _buffPtr;
  
    //! blowfish key
    BF_KEY _bfKey;
    
    //! indicates if stream is open or not
    bool _open;
    

  };  // class OBlowfishStream

};  // namespace MynahSA

#endif
