/************************************************************************************
 *    This file is part of the MynahSA streaming and archiving toolkit              *
 *    Copyright (C) 2006 Mynah-Software Ltd. All Rights Reserved.                   *
 *                                                                                  *
 *    This program is free software; you can redistribute it and/or modify          *
 *    it under the terms of the GNU General Public License, version 2               *
 *    as published by the Free Software Foundation.                                 *
 *                                                                                  *
 *    This program is distributed in the hope that it will be useful,               *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of                *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                 *
 *    GNU General Public License for more details.                                  *
 *                                                                                  *
 *    You should have received a copy of the GNU General Public License along       *
 *    with this program; if not, write to the Free Software Foundation, Inc.,       *
 *    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.                   *
 *                                                                                  *
 ************************************************************************************/

#ifndef __o_ssl_stream_hpp
#define __o_ssl_stream_hpp

#include <iostream>

#include <stdexcept>

#include <string>

#include <openssl/ssl.h>

#include "ostreambase.hpp"


namespace MynahSA { 
  /** class OSSLStream is an output stream designed to place the builtin C++ types onto an SSL stream
  *    The class operates by calling SSL_write(...) once for each type.
  *
  *    This class does not take ownership of the SSL object passed into it.  The user is responsible
  *    for correct cleanup of the SSL, and furthermore, ensuring that isntances of OSSLStream do not
  *    use the SSL* after it is closed!
  *
  */
  class OSSLStream : public OStreamBase {
  public:
    //! constructor - takes an input stream
    OSSLStream(SSL* s);
  
    //! copy constructor - needed to work around unspecified copy constructor behavior in SUNPro c++
    OSSLStream(const OSSLStream& s);
  
    //! destructor - note: SSL is not freed
    ~OSSLStream();
  
  
    virtual OStreamBase& operator<<(const char);
    virtual OStreamBase& operator<<(const unsigned char);
    
    virtual OStreamBase& operator<<(const short);
    virtual OStreamBase& operator<<(const unsigned short);
    
    virtual OStreamBase& operator<<(const bool);
    
    virtual OStreamBase& operator<<(const int);
    virtual OStreamBase& operator<<(const unsigned int);
    
    virtual OStreamBase& operator<<(const long long);
    virtual OStreamBase& operator<<(const unsigned long long);
    
    virtual OStreamBase& operator<<(const double);
    virtual OStreamBase& operator<<(const float);
    
    //! output streaming of std strings
    virtual OStreamBase& operator<<(const std::string& s);
      
    // using declaration to allow base method to be used.
    using OStreamBase::operator<<;
      
    //! force the internal buffers to be flushed
    void flush();
      
  private:
    //! SSL pointer - the transport layer interface
    SSL* _ssl;
    
    //! local buffer
    char* _buffer;
    
    //! buffer pointer - points to the current back of the buffer
    unsigned int _buffPtr;
  
    //! SSL_write replacement - performs buffer management
    int sslWrite(const void* data, unsigned int size);
    
  };
};



#endif
