/************************************************************************************
 *    This file is part of the MynahSA streaming and archiving toolkit              *
 *    Copyright (C) 2006 Mynah-Software Ltd. All Rights Reserved.                   *
 *                                                                                  *
 *    This program is free software; you can redistribute it and/or modify          *
 *    it under the terms of the GNU General Public License, version 2               *
 *    as published by the Free Software Foundation.                                 *
 *                                                                                  *
 *    This program is distributed in the hope that it will be useful,               *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of                *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                 *
 *    GNU General Public License for more details.                                  *
 *                                                                                  *
 *    You should have received a copy of the GNU General Public License along       *
 *    with this program; if not, write to the Free Software Foundation, Inc.,       *
 *    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.                   *
 *                                                                                  *
 ************************************************************************************/

#ifndef __o_text_stream_hpp
#define __o_text_stream_hpp

#include <iostream>

#include "ostreambase.hpp"

namespace MynahSA { 

  /** OTextStream binds a set of output stream operators onto a standard ostream.  The
  *  purpose of this class is to provide space-separated formatting of the output data.
  *  This means that characters and unsigned characters are encoded as integers and 
  *  written onto the stream.  From looking at the output data it will be difficult 
  *  (if not impossible) to identify which numbers represent characters, shorts or 
  *  integers.
  *
  *  Notes: This stream format stores floating point types using the precision
  *  specification of the stream passed in during class construction.  For some
  *  applications the text representation of floating point numbers may not provide
  *  enough precision.
  */
  class OTextStream : public OStreamBase { 
  public:
    //! constructor - build on top of a std::ostream
    OTextStream(std::ostream& o);
    
    //! copy constructor
    OTextStream(const OTextStream& os);
    
    //! virtual destructor
    virtual ~OTextStream();
  
    //! specialization to prevent direct character interpretation
    virtual OStreamBase& operator<<(const char);
    
    //! specialization to prevent direct character interpretation
    virtual OStreamBase& operator<<(const unsigned char);
    
    //! ostream short
    virtual OStreamBase& operator<<(const short);
    
    //! ostream unsigned short
    virtual OStreamBase& operator<<(const unsigned short);
    
    //! ostream int
    virtual OStreamBase& operator<<(const int);
    
    //! ostream unsigned int
    virtual OStreamBase& operator<<(const unsigned int);
    
    //! ostream bool
    virtual OStreamBase& operator<<(const bool);
    
    //! ostream long long
    virtual OStreamBase& operator<<(const long long);
    
    //! ostream unsigned long long
    virtual OStreamBase& operator<<(const unsigned long long);
    
    //! ostream double
    virtual OStreamBase& operator<<(const double);
    
    //! ostream float
    virtual OStreamBase& operator<<(const float);
    
    //! we provide an implementation for strings, but user classes may override.
    virtual OStreamBase& operator<<(const std::string&);
  
  private:
    std::ostream& _os;
  
  };
};
#endif
