/************************************************************************************
 *    This file is part of the MynahSA streaming and archiving toolkit              *
 *    Copyright (C) 2006 Mynah-Software Ltd. All Rights Reserved.                   *
 *                                                                                  *
 *    This program is free software; you can redistribute it and/or modify          *
 *    it under the terms of the GNU General Public License, version 2               *
 *    as published by the Free Software Foundation.                                 *
 *                                                                                  *
 *    This program is distributed in the hope that it will be useful,               *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of                *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                 *
 *    GNU General Public License for more details.                                  *
 *                                                                                  *
 *    You should have received a copy of the GNU General Public License along       *
 *    with this program; if not, write to the Free Software Foundation, Inc.,       *
 *    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.                   *
 *                                                                                  *
 ************************************************************************************/

#ifndef __sp_impl_hpp
#define __sp_impl_hpp

#ifdef MYNAHSA_USE_BOOST

#include <boost/shared_ptr.hpp>
#define SHARED_PTR boost::shared_ptr
#define STATIC_PTR_CAST boost::static_pointer_cast

#else

#define SHARED_PTR MynahSA::SharedPtr  
#define STATIC_PTR_CAST MynahSA::StaticPtrCast

#include <mynahsa/thread.hpp>

namespace MynahSA { 

  namespace Detail { 
    class SPBase { 
    public:
      inline SPBase() : _ptr(0), _count(0) { }
      inline SPBase(void* p) : _ptr(p), _count(0) { }
      inline virtual ~SPBase() { 
      }
    
      inline void ref() { 
        Lock l(_m);
        _count++;
      }
    
      inline void unref() { 
        Mutex::lock(_m);
        if (--_count == 0) { 
          // delete the object
          deleteMe();
          
          // unlock the mutex
          Mutex::unlock(_m);
          
          // delete us (which implicitly deletes the mutex)
          delete this;
          
        } else { 
          Mutex::unlock(_m);
        }
      }

      inline void* get() const { return _ptr; }
    
    protected:
      virtual void deleteMe() = 0;
    
      void* _ptr;
    
    private:
      unsigned int _count;
      
      Mutex _m;    
    };
    
    template <class L>
    class SPImpl : public SPBase { 
    public:
      inline SPImpl(L* p) : SPBase(p) { 
        ref();
      }
    
      inline ~SPImpl() { 
      }
      
    protected:
      void deleteMe() { 
        delete static_cast<L*>(_ptr);
        _ptr = 0;
      }
    };
  
  };

  /** Class SharedPtr is a shared pointer implementation for MynahSA.  Shared poitners are used
   *  in place of standard c pointers to prevent the ills that come along with misues of standard
   *  c style pointers.  
   *
   *  This shared pointer implementation uses template specialization and an underlying container
   *  to allow conversions to and from void types and up and down class hierarchies.  The 
   *  conversions are performed using the function StaticPtrCast.  Casting a pointer from one
   *  type to another preserves the reference count.  
   *
   *  When all holders of a shared pointer are deleted the object held by the shared pointer is
   *  also deleted.
   */
  template<class T>
  class SharedPtr { 
  public:
    template <class B,class A>
    friend SharedPtr<B> StaticPtrCast(const SharedPtr<A>& src);
  public:
    //! default constructor - creates a null pointer
    SharedPtr() : _spref(0) {
    }
      
    
    /**
     * Constructor - creates a new shared pointer of type T, along with pointer carrier and reference
     *  count.  Initially the reference count is set to 1.
     * @param p - The pointer that will now be owned by this shared pointer instance.
     * @return 
     */
    SharedPtr(T* p) : _spref(new Detail::SPImpl<T>(p)) { 
    }
      
    //! copy constructor
    SharedPtr(const SharedPtr& p) : _spref(p._spref) {
      if (_spref) { 
        // up the reference count
        _spref->ref();
      }
    }
      
    
    /**
     * This is a compiler workaround constructor.  Some compilers have issues dealing with friendship
     * relations between instances with different template parameters.  This constructor is designed as
     * a work-around by passing in the pointer carrier.  It is not intended for general use.
     * @param p - a pointer carrier.
     * @return 
     */
    SharedPtr(Detail::SPBase* p) : _spref(p) {
      if (p) { 
        p->ref();
      }
    }
    
    //! Destructor - destroyes instance if reference count goes to zero.
    virtual ~SharedPtr() { 
      if (_spref) { 
        // drop reference count, deleting object if object is no longer referenced
        _spref->unref();
      }
    }
      
    //! Provides direct access to the underlying pointer
    T* get() const {
      if (_spref){ 
        return static_cast<T*>(_spref->get());
      } else { 
        return 0;
      }
    }
    
    //! dereference operator.
    T& operator*() const { 
      // note: this is a failure if pointer is zero! A throw should be placed here in the event that pointer is zero
      return *(static_cast<T*>(_spref->get()));
    }
  
    //! pointer dereference operator
    T* operator->() const { 
      return static_cast<T*>(_spref->get());
    }
  
    //! LT comparable.  Allows sorting of pointer objects
    bool operator<(const SharedPtr& in) const { 
      return _spref->get() < in._spref->get();
    }
  
    //! Assignment - allows copying of a shared pointer while preserving S.P. behavior
    SharedPtr& operator=(const SharedPtr& in) { 
      if (_spref) { 
        _spref->unref();
      }
      _spref = in._spref;
      // Note: Don't increment spref if null pointer!
      if (_spref) { 
        _spref->ref();
      }
  
      return *this;
    }
    
    //! Shared pointer equality test.
    bool operator==(const SharedPtr& in) const { 
      return _spref == in._spref;
    }
      
    //! Shared pointer equality to base pointer (generally used for zero test).
    bool operator==(T* ptr) const { 
      //  is    (spref 0)    yes                no
      return _spref == 0 ? (ptr == 0) : ptr == static_cast<T*>(_spref->get());
    }
      
    //! cast to bool - true if pointer is not zero.
    operator bool() const { 
      return _spref == 0 ? false : _spref->get() != 0;
    }
      
    //! Pointer conversion operator.  Allows type conversion of shared pointers.
    template<class B>
    operator SharedPtr<B>() const { 
      // invalid pointer, cast zero to target
      if (!_spref->get()) { 
        return SharedPtr<B>();
      } 
      
      static_cast<B*>( static_cast<T*>(_spref->get()) ); // this is an assert - that static casting is allowed
      SharedPtr<B> dst(_spref);
      return dst;
    }
      
      
  private:
    //! pointer carrier pointer!
    Detail::SPBase* _spref;
  };
  
  
  
  /** This is a specialization for void pointers.  The difference between the generic SharedPtr and
   *  the void specialization is that void shared pointers have no template dereference operator.
   *  This means that you cannot perform the following:
   *  SharedPtr<void> spv( ... );
   *  *spv;           // error, attempt to derference a void pointer
   */
  template<>
  class SharedPtr<void> { 
    //template<class B>
    //friend SharedPtr<void> StaticPtrCast(const SharedPtr<B>&);
  
  
    template <class B,class A>
    friend SharedPtr<B> StaticPtrCast(const SharedPtr<A>& src);
  
  
  public:
    //! default constructor
    SharedPtr() : _spref(0) {
    }
  
    // cannot construct a void instance
      
    //! copy constructor
    SharedPtr(const SharedPtr& p) : _spref(p._spref) {
      if (_spref) { 
        // up the reference count
        _spref->ref();
      }
    }
      
      
    SharedPtr(Detail::SPBase* p) : _spref(p) {
      if (p) { 
        p->ref();
      }
    }
      
    virtual ~SharedPtr() { 
      if (_spref) { 
        // drop reference count, deleting object if object is no longer referenced
        _spref->unref();
      }
    }
      
    //! access to pointer
    void* get() const {
      if (_spref){ 
  
        return _spref->get();
      } else { 
        return 0;
      }
    }
    
    // no reference method
  
    bool operator<(const SharedPtr& in) const { 
      return _spref->get() < in._spref->get();
    }
  
    
    SharedPtr& operator=(const SharedPtr& in) { 
      if (_spref) { 
        _spref->unref();
      }
      _spref = in._spref;
      _spref->ref();
  
      return *this;
    }
    
    
    bool operator==(const SharedPtr& in) const { 
      return _spref == in._spref;
    }
      
    bool operator==(void* ptr) const { 
      //  is    (spref 0)    yes                no
      return _spref == 0 ? (ptr == 0) : ptr == static_cast<void*>(_spref->get());
    }
      
    operator bool() const { 
      return _spref == 0 ? false : _spref->get() != 0;
    }
    
    template<class B>
    operator SharedPtr<B>() const { 
      // invalid pointer, cast zero to target
      if (!_spref->get()) { 
        return SharedPtr<B>();
      } 
      // don't need the assertion, we know it will succeed already      
      SharedPtr<B> dst(_spref);
      return dst;
    }
    
      
      
  private:
    Detail::SPBase* _spref;
  };
  
  template <class B,class A>
  SharedPtr<B> StaticPtrCast(const SharedPtr<A>& src) {
    static_cast<B*>( static_cast<A*>(0)); // this is an assert - that static casting is allowed
    SharedPtr<B> dst(src._spref);
    return dst;
  }
  



};

#endif

#endif
