/************************************************************************************
 *    This file is part of the MynahSA streaming and archiving toolkit              *
 *    Copyright (C) 2006 Mynah-Software Ltd. All Rights Reserved.                   *
 *                                                                                  *
 *    This program is free software; you can redistribute it and/or modify          *
 *    it under the terms of the GNU General Public License, version 2               *
 *    as published by the Free Software Foundation.                                 *
 *                                                                                  *
 *    This program is distributed in the hope that it will be useful,               *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of                *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                 *
 *    GNU General Public License for more details.                                  *
 *                                                                                  *
 *    You should have received a copy of the GNU General Public License along       *
 *    with this program; if not, write to the Free Software Foundation, Inc.,       *
 *    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.                   *
 *                                                                                  *
 ************************************************************************************/


#ifndef __ssl_connection_manager_hpp
#define __ssl_connection_manager_hpp

#include <openssl/ssl.h>
#include <openssl/err.h>

#include <mynahsa/requestresponsehandler.hpp>

#include <mynahsa/streamconstructor.hpp>

#include <mynahsa/sslconnectionverifier.hpp>

namespace MynahSA { 
  
  /** class SSLConnectionManager - This class is designed to manage an individual SSL connection.
  *
  *  Design:  SSLServer manages ALL incoming connections.  When an incomming connection is requested
  *    the SSLServer spawns a thread to manage all I/O over that particular SSL connection.  The class
  *    responsible for handling the connection is SSLConnectionManager.  SSLConnectionManager performs
  *    some basic stream initialization, then proceeds to stream objects off of the SSL stream and 
  *    reconstruct them in memory.  Once the in-memory image is ready, the request/response handler
  *    is invoked to transform an object received off of the stream into a response.  The response is then
  *    placed back onto the stream for the client to process.
  *
  *  Notes: 
  *    SSLConnection manager and descendant classes must be thread safe.
  *    StreamType must support copy construction.
  */
  class SSLConnectionManager { 
  public:
    //! constructor - hand the constructor a request handler object
    SSLConnectionManager(RequestResponseHandler& handler,
                         const StreamConstructor& cons,
                         SSLConnectionVerifier& verifier);
  
    ~SSLConnectionManager();
  
    
    /** Operator - this function is invoked when a new connection is established.  The SSL connection is passed
     *  in, and SSL_accept is performed inside this function.
     * 
     * @param ssl - the SSL connection.
     */
    void operator()(SSL* ssl);
  
  private:
  
    //! pointer to response handler object
    RequestResponseHandler& _handler;
    
    //! pointer to stream object constructor
    const StreamConstructor& _streamConstructor;
    
    //! the verifier object is used to determine whether an incoming connection should be allowed or not.
    SSLConnectionVerifier& _verifier;
  };
};

#endif
