/************************************************************************************
 *    This file is part of the MynahSA streaming and archiving toolkit              *
 *    Copyright (C) 2006 Mynah-Software Ltd. All Rights Reserved.                   *
 *                                                                                  *
 *    This program is free software; you can redistribute it and/or modify          *
 *    it under the terms of the GNU General Public License, version 2               *
 *    as published by the Free Software Foundation.                                 *
 *                                                                                  *
 *    This program is distributed in the hope that it will be useful,               *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of                *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                 *
 *    GNU General Public License for more details.                                  *
 *                                                                                  *
 *    You should have received a copy of the GNU General Public License along       *
 *    with this program; if not, write to the Free Software Foundation, Inc.,       *
 *    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.                   *
 *                                                                                  *
 ************************************************************************************/

#ifndef __ssl_server_h
#define __ssl_server_h

#include <string>
#include <stdexcept>

#include <openssl/ssl.h>
#include <openssl/err.h>

#include <mynahsa/sslconnectionmanager.hpp>

#include <mynahsa/exceptionbase.hpp>


#define ENABLE_THREADED_SERVER

namespace MynahSA { 
  /** Class SSLServer is used to handle root connections - e.g. one instance of this class will handle 
  *  incoming connections on a TCP port.  Once the connection is established, the connection is handed to
  *  a ConnectionManager object - the template parameter used here takes the instance of type T and uses 
  *  its operator() method to handle the individual connection.
  *  <br><br>
  *  Note: class T must provide an void operator()(SSL*) method - the operator will be invoked when
  *  a connection has been established.
  *        
  */
  class SSLServer {
  public:
    /** SSLServer constructor - takes a server object - an object that is required for dealing with SSL requests<br>
    *  cFile - a file name for the SSL certificate file<br>
    *  kFile - a file name for the SSL private key<br>
    *  port - the port to answer requests on
    */
    SSLServer(SSLConnectionManager& sobj, 
              std::string cFile, 
              std::string kFile, 
              int port,
              std::string caFile="" );
  
    /** in memory loading of SSL certificate and SSL RSA private key.  Both
    *  must be in ASN1 format.
    *
    *  Note: this constructor is untested
    */
    SSLServer(SSLConnectionManager& sobj, 
              X509* certp, 
              EVP_PKEY* pkeyp, 
              int port,
              std::string caFile="");
  
  
    //! close port and cleanup information
    ~SSLServer();
          
    /** Check for new connections, block for wait_t amount of seconds,
    *  if no connection is made within wait_t function returns.  This must
    *  be called regularly from within main 
    */
    void checkClients(int wait_t=0);
  
  private:
  
    
    /** Load certificates from a file
     * 
     * @param cFile - certificate file
     * @param kFile - private key file
     */
    void loadCerts(std::string cFile, std::string kFile);
  
    //! Load algorithms and create context.
    void createCTX(std::string serverCAFile);
  
    //! Create and attach socket.
    void bindPort(void);
  
  
    //! SSL context
    SSL_CTX *_ctx;
  
    //! port server is to operate on
    int _port;
  
    //! FD for server
    int _master;
    
    //! server object
    SSLConnectionManager& _serverObject;
  };  
};

#endif
