/************************************************************************************
 *    This file is part of the MynahSA streaming and archiving toolkit              *
 *    Copyright (C) 2006 Mynah-Software Ltd. All Rights Reserved.                   *
 *                                                                                  *
 *    This program is free software; you can redistribute it and/or modify          *
 *    it under the terms of the GNU General Public License, version 2               *
 *    as published by the Free Software Foundation.                                 *
 *                                                                                  *
 *    This program is distributed in the hope that it will be useful,               *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of                *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                 *
 *    GNU General Public License for more details.                                  *
 *                                                                                  *
 *    You should have received a copy of the GNU General Public License along       *
 *    with this program; if not, write to the Free Software Foundation, Inc.,       *
 *    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.                   *
 *                                                                                  *
 ************************************************************************************/

#ifndef __tcp_rpc_client_hpp
#define __tcp_rpc_client_hpp

#include "iobase.hpp"

#ifdef WIN32
#include <windows.h>
#include <winsock2.h>
#else
#include <netinet/in.h>
#include <arpa/inet.h>
#endif

#include "tcparchivestream.hpp"
#include "streamconstructor.hpp"

#include <string>

#include <mynahsa/spimpl.hpp>


namespace MynahSA { 
  /** TCPRPCCLient is a client class for handling RPC requests.  Configure this
  *  class with a IP address and port in the constructor - these are then used
  *  in the RPC method for TCP level communications.
  * 
  */
  class TCPRPCClient {
  public:
    /** constructor - takes parameters:
     *   @param sc - a stream constructor object.  This is necessary for reconstruction of objects from the TCP Stream
     *   @param ipaddress - an integer packed IPv4 internet address for the server
     *   @param port - the port number of the server
     *   @param certificateFile - A client specific certificate file (optional parameter).
     *   @param privateKeyFile  - The private key for the certificate.
     */
    TCPRPCClient(const StreamConstructor& sc,
                 unsigned int ipaddress, 
                 unsigned int port);
    
    
    /** constructor - takes parameters:
     *   @param sc - a stream constructor object.  This is necessary for reconstruction of objects from the TCP Stream
     *   @param machineName - the string name of the machine running a server
     *   @param port - the port number of the server
     *   @param certificateFile - A client specific certificate file (optional parameter).
     *   @param privateKeyFile  - The private key for the certificate.
     */
    TCPRPCClient(const StreamConstructor& sc,
                 std::string machineName, 
                 unsigned int port);
    
    //! destructor
    ~TCPRPCClient();
  
    //! handler for performing the rpc call
    SHARED_PTR<IoBase> rpc(SHARED_PTR<IoBase> req);
  
    //! return true if the connection is still open
    bool is_open() const { return _isopen; }
  
    //! close and terminate the connection
    void close();
  
  
  private:
    //! complete the work of the constructor
    void init(const StreamConstructor& sc,
              unsigned int ipaddress,
              unsigned int port);
  
    //! set to true if connection is open; reset to false on error
    bool _isopen;
    
    //! socket descriptor
    int         _sd;
  
    //! the underlying stream that we use for rpc communication
    SHARED_PTR<TCPArchiveStream> _stream;
  
  };
};

#endif
