/************************************************************************************
 *    This file is part of the MynahSA streaming and archiving toolkit              *
 *    Copyright (C) 2006 Mynah-Software Ltd. All Rights Reserved.                   *
 *                                                                                  *
 *    This program is free software; you can redistribute it and/or modify          *
 *    it under the terms of the GNU General Public License, version 2               *
 *    as published by the Free Software Foundation.                                 *
 *                                                                                  *
 *    This program is distributed in the hope that it will be useful,               *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of                *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                 *
 *    GNU General Public License for more details.                                  *
 *                                                                                  *
 *    You should have received a copy of the GNU General Public License along       *
 *    with this program; if not, write to the Free Software Foundation, Inc.,       *
 *    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.                   *
 *                                                                                  *
 ************************************************************************************/

#ifndef __thread_hpp
#define __thread_hpp

#ifdef MYNAHSA_USE_BOOST
#include <boost/thread.hpp>
#include <boost/thread/mutex.hpp>
namespace MynahSA { 
  typedef boost::mutex Mutex;
  typedef boost::mutex::scoped_lock Lock;
};

#else

#ifndef WIN32
#include <pthread.h>
#else
#include <windows.h>
#endif


#include <iostream>


namespace MynahSA { 
  /** Class Mutex implements mutual exclusion using the Posix pthread library or using Win32's 
   *  mutex facilities.
   *
   *  The two static methods provided are used to lock and unlock the mutex.
   *  Destroying this mutex while it is locked causes undefined behavior - you should never do this!
   *  See the man pages for pthread_mutex_unlock on Linux for more details.
   *
   *  Note: Class Mutex is unavailable when MYNAHSA_USE_BOOST is set as a compile time option.
   *
   */
  class Mutex { 
    friend class Lock;
  public:
    //! Default constructor 
    Mutex();
    //! Default destructor - call only when mutex is unlocked!
    ~Mutex();
  
    //! Lock a specific mutex - this blocks until the mutex is available
    inline static void lock(Mutex& m) { 
#ifdef WIN32
      WaitForSingleObject(m._mutex, INFINITE); // obtain lock
#else
      pthread_mutex_lock(&m._mutex); 
#endif
    }
    
    //! unlock a specific mutex
    inline static void unlock(Mutex& m) { 
#ifdef WIN32
      ReleaseMutex(m._mutex);
#else
      pthread_mutex_unlock(&m._mutex); 
#endif
    }
  
  private:
    //! non-copyable - this class cannot be copied
    Mutex(const Mutex&) { }
#ifdef WIN32
    //! handle for win32 (Only available on win32)
    HANDLE _mutex;
#else
    // posix mutex (only available on posix machines)
    pthread_mutex_t _mutex;
#endif
  };

  /** Class Lock is a convienence class.  It is created with a reference to a specific mutex,
   *  and that mutex is locked during Lock construction.  E.g. Once you have created a lock, you
   *  hold the mutex.  When a lock is destroyed (by means of the destructor) the mutex is freed.
   *
   *  The usage pattern for class Lock is a single function and/or frame in which the critical
   *  section behavior is contained.  Consider:
   *  void f() {
   *    Lock l(myMutex);
   *    ... do shared stuff ...
   *  }
   *  The end of the frame releases the lock.  This usage pattern can help prevent faults where
   *  locks are left in the acquired state.
   *
   *  Note: Class Lock is unavailable when MYNAHSA_USE_BOOST is set as a compile time option.
   *
   */
  class Lock { 
  public:
    //! constructor - requires a reference to a mutex object, blocks until mutex is acquired
    Lock(Mutex& m);
    //! destructor - releases the mutex
    ~Lock();
  private:
    //! non-copyable
    Lock(const Lock& l ) : _mutex(l._mutex) { }
    //! mutex reference
    Mutex& _mutex;
  };
  

  /** Class thread implements a thread object.  The template typename must specify a functor object.
   *
   *  The functor object must support copy construction: The Thread class copies the object to a heap
   *  allocated object, then spawns a thread on it.  This allows the original object to go out of scope
   *  without causing segmentation faults.  
   *
   *  The copied object is deleted when the thread exists.
   *  
   *  Once a Thread instance is created an operating system thread is created and the operator() method
   *  of the provided class reference is invoked.  
   *
   *  Parameters are not passable to the thread using the construction mechanism - rather instead,
   *  all parameters should be passed in using the functor's constructor.
   *
   *  Threads may be joined using the join method.  Calling join on a thread from the parent thread
   *  will cause the parent thread to block until the child has terminated.
   *
   *  Note: Class Thread is unavailable when MYNAHSA_USE_BOOST is set as a compile time option.
   *
   * \bug no handling of thread limits yet.
   * \bug no handling of potential failures in thread creation.
   */
  template<class T>
  class Thread { 
  public:
    //! Cosntructor - pass in a functional object
    Thread(const T& startclass) {
      T* copy = new T(startclass);  // create a startclass copy on the heap
      
#ifndef WIN32
      pthread_create(&_thread,
                     0,
                     Thread<T>::startAddr,
                     copy);
#else 
      _hThread = CreateThread(0,                    // LPSECURITY_ATTRIBUTES
                              0,                    // dwStackSize - 0 means same as parent
                              (LPTHREAD_START_ROUTINE) Thread<T>::startAddr, // start address
                              copy,        // data (this ptr)
			                        0,
			                        &_thread);    
#endif
    }
    /** Destructor: In the event of the thread being thrown away, we detach, allowing
     *    memory to be recovered on thread termination.
     */
    ~Thread() { 
#ifndef WIN32
      pthread_detach(_thread);
#endif
    }    
    /** Join a thread.  On Win32 will return 0 by default.  On unix will return whatever the 
     *  called thread returned.
     */
    void* join() { 
#ifdef WIN32
      WaitForSingleObject( _hThread, INFINITE );
      return 0;
#else
      void* vrtn;
      pthread_join(_thread,&vrtn);
      return vrtn;
#endif
    }
      
  private:
    //! static bounce function for OS thread
    static void* startAddr(void* args) {
      T* tptr = reinterpret_cast<T*>(args);
      (*tptr)();
      delete tptr;
      return 0;
    }
  
#ifdef WIN32
    //! WIN32 only - handle to thread
    HANDLE _hThread;
    //! WIN32 only - thread id
    DWORD _thread;
#else  
    //! Posix only - thread struct.
    pthread_t _thread;
#endif

  };
};
#endif

#endif
