/************************************************************************************
 *    This file is part of the MynahSA streaming and archiving toolkit              *
 *    Copyright (C) 2006 Mynah-Software Ltd. All Rights Reserved.                   *
 *                                                                                  *
 *    This program is free software; you can redistribute it and/or modify          *
 *    it under the terms of the GNU General Public License, version 2               *
 *    as published by the Free Software Foundation.                                 *
 *                                                                                  *
 *    This program is distributed in the hope that it will be useful,               *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of                *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                 *
 *    GNU General Public License for more details.                                  *
 *                                                                                  *
 *    You should have received a copy of the GNU General Public License along       *
 *    with this program; if not, write to the Free Software Foundation, Inc.,       *
 *    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.                   *
 *                                                                                  *
 ************************************************************************************/

#include <mynahsa/iblowfishstream.hpp>
    
#ifndef WIN32
#include <strings.h>
#else
#define bzero(thing, sz) memset((thing), 0, (sz));
#endif

#if BYTE_ORDER == BIG_ENDIAN
#include <mynahsa/byteswap.hpp>
#endif

using namespace std;

namespace MynahSA { 
  IBlowfishStream::IBlowfishStream(istream& is, const string& key) : _is(is), 
                                                                     _buffPtr(8) {
    // NOTE: We initialize _buffPtr to 8 - this forces a fill call on first read.
                                                                     
    bzero(_inBuffer, 8);
    bzero(_outBuffer, 8);
    bzero(_ivec, 8);
    bzero(&_bfKey, sizeof(BF_KEY)); // zero the blowfish key
                                                                             
    // initialize blowfish key                                                                             
    BF_set_key(&_bfKey, key.size(), (const unsigned char*) key.c_str());
                                       
  }

  IBlowfishStream::IBlowfishStream(const IBlowfishStream& is) : _is(is._is),
                                                                _buffPtr(is._buffPtr) {
    // copy parameters
    memcpy(_inBuffer, is._inBuffer, 8);
    memcpy(_outBuffer, is._outBuffer, 8);
    memcpy(_ivec, is._ivec, 8);
    memcpy(&_bfKey, &is._bfKey, sizeof(BF_KEY));
    
  }                                                               

  IBlowfishStream::~IBlowfishStream() { 
    // nothing to do here
  }
  
  void IBlowfishStream::fillBuffer() { 
    // read data from stream
    _is.read( (char*)_inBuffer, 8);
    
    // did we read the correct amount of data?
    if (_is.gcount() != 8) { 
      throw IStreamReadError("IBlowfishStream::fillBuffer - short read!");
    }
    
    // now decrypt the buffer and set _buffPtr to zero
    BF_cbc_encrypt(_inBuffer,
                   _outBuffer,
                   8,
                   &_bfKey,
                   _ivec,
                   BF_DECRYPT);
                   
    // reset _buffPtr
    _buffPtr=0;
  }

  void IBlowfishStream::get(char& c) {
    if (_buffPtr >= 8) { 
      fillBuffer();
    }
    c = _outBuffer[_buffPtr++];
  }
  
#if BYTE_ORDER == LITTLE_ENDIAN
#define MAKE_BF_LOADER( type )\
  IStreamBase& IBlowfishStream::operator>>(type & value) {\
    char* cp = reinterpret_cast<char*>(&value);\
    for (unsigned int i=0; i<sizeof( type ); i++) {\
      get(cp[i]); \
    }\
    return *this; \
  }
  
#else

#define MAKE_BF_LOADER( type )\
  IStreamBase& IBlowfishStream::operator>>(type & value) {\
    char* cp = reinterpret_cast<char*>(&value);\
    for (unsigned int i=0; i<sizeof( type ); i++) {\
      get(cp[i]); \
    }\
    byteswap( value ); \
    return *this; \
  }
#endif
  
  MAKE_BF_LOADER(char);
  MAKE_BF_LOADER(unsigned char);
  MAKE_BF_LOADER(short);
  MAKE_BF_LOADER(unsigned short);
  MAKE_BF_LOADER(int);
  MAKE_BF_LOADER(unsigned int);
  MAKE_BF_LOADER(long long);
  MAKE_BF_LOADER(unsigned long long);
  MAKE_BF_LOADER(double);
  MAKE_BF_LOADER(float);  
#undef MAKE_BF_LOADER  
  // this is a code workaround for architectures that use 4 byte bool specifications.  This clamps it onto a 1 
  // byte specification which is like x86_64.  After all there is no need to spend more that 1 bit on a bool anyway!
  IStreamBase& IBlowfishStream::operator>>(bool & b ) { 
    char c;
    (*this) >> c;
    b = c;
    return *this;
  }


  
  
};
