/************************************************************************************
 *    This file is part of the MynahSA streaming and archiving toolkit              *
 *    Copyright (C) 2006 Mynah-Software Ltd. All Rights Reserved.                   *
 *                                                                                  *
 *    This program is free software; you can redistribute it and/or modify          *
 *    it under the terms of the GNU General Public License, version 2               *
 *    as published by the Free Software Foundation.                                 *
 *                                                                                  *
 *    This program is distributed in the hope that it will be useful,               *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of                *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                 *
 *    GNU General Public License for more details.                                  *
 *                                                                                  *
 *    You should have received a copy of the GNU General Public License along       *
 *    with this program; if not, write to the Free Software Foundation, Inc.,       *
 *    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.                   *
 *                                                                                  *
 ************************************************************************************/

#include <mynahsa/isslstream.hpp>

#if BYTE_ORDER == BIG_ENDIAN
#include <mynahsa/byteswap.hpp>
#endif



using namespace std;



namespace MynahSA { 

  //! constructor - takes an input stream
  ISSLStream::ISSLStream(SSL* s) : _ssl(s), _open(true) {
  }
  
  ISSLStream::ISSLStream(const ISSLStream& i) : _ssl(i._ssl), _open(i._open) { 
  }
  
  //! destructor - note: SSL is not freed
  ISSLStream::~ISSLStream() {
  }
  
  int ISSLStream::sslRead(void* buf, int num) { 
        
    // wrapper for SSL_read; takes care of zero length read - which is not an error, just a closed stream
    // archive
    
    char* cbuf = reinterpret_cast<char*>(buf);
    
    bool done = false;
    
    int originalNum  = num;  // copy of original number of bytes to read
    
    while (!done) { 
      int result = SSL_read(_ssl, cbuf, num);
      int errorCode = SSL_get_error(_ssl, result);
      
      // treat error condition
      if (errorCode == SSL_ERROR_ZERO_RETURN) { 
        throw IStreamEOF();
      } else if (errorCode != 0) { 
        throw IStreamReadError("ISSLStream::sslRead An error occured in stream read");
      } // otherwise, error code is zero - normal
      
      if (result < num) { 
        // handle a short read; this occurs when a write spans multiple SSL records
        cbuf += result;
        num -= result;
      } else { 
        num -= result;
        // num should be zero here
        done = true;
      }
    }
    // assertion check - num should be zero
    
    return originalNum;
  }
  
  
#if BYTE_ORDER == LITTLE_ENDIAN

#define MAKE_SSL_READER( t ) \
IStreamBase& ISSLStream::operator>>( t & v ) { \
  if (sslRead( &v, sizeof( t ) ) != sizeof ( t )) { \
    _open = false; \
    throw IStreamReadError("ISSLStream::operator>>( " #t " ) - short read"); \
  } \
  return *this; \
}

#else

#define MAKE_SSL_READER( t ) \
IStreamBase& ISSLStream::operator>>( t & v ) { \
  if (sslRead( &v, sizeof( t ) ) != sizeof ( t )) { \
    _open = false; \
    throw IStreamReadError("ISSLStream::operator>>( " #t " ) - short read"); \
  } \
  byteswap( v ); \
  return *this; \
}

#endif
  
  
  MAKE_SSL_READER( char );
  MAKE_SSL_READER( unsigned char );
  MAKE_SSL_READER( short );
  MAKE_SSL_READER( unsigned short );
  MAKE_SSL_READER( int );
  MAKE_SSL_READER( unsigned int );
  MAKE_SSL_READER( long long );
  MAKE_SSL_READER( unsigned long long );
  MAKE_SSL_READER( double );
  MAKE_SSL_READER( float );
  
#undef MAKE_SSL_READER
  // this is a code workaround for architectures that use 4 byte bool specifications.  This clamps it onto a 1 
  // byte specification which is like x86_64.  After all there is no need to spend more that 1 bit on a bool anyway!
  IStreamBase& ISSLStream::operator>>(bool & b ) { 
    char c;
    (*this) >> c;
    b = (c ? true : false);  // work around valgrind hit
    return *this;
  }
  
  
  //! input streaming of std strings
  IStreamBase& ISSLStream::operator>>(string& s) {
    int size=0;
    char* buffer=0;
  
    // recover the size parameter
    (*this) >> size;
  
    if (size == 0) { 
      // early terminate of empty strings
      s = "";
      return *this;
    }
  
    // allocate temporary buffer
    buffer = new char[size+1];
    if (!buffer) { 
      throw IStreamReadError("Buffer memory-allocation failed in std::string reconstruction");
    }
    
    // receive the buffer
    if (sslRead(buffer,size) != size) { 
      delete []buffer;
      _open = false;
      throw IStreamReadError("Short read in ISSLStream on std::string reconstruction");
    }
  
    // null terminate the string - this is because std::string will read the buffer as a c string
    buffer[size] = '\0';
  
    // copy buffer to string
    s = string(buffer);
    
    // delete buffer
    delete []buffer;
    
    return *this;
  }
  
  void ISSLStream::get(char& c) {
    int result = sslRead(&c, sizeof(char) );
    if (result != sizeof(char)) { 
      throw IStreamReadError("ISSLStream::get - short read");
    }
  }
};
