/************************************************************************************
 *    This file is part of the MynahSA streaming and archiving toolkit              *
 *    Copyright (C) 2006 Mynah-Software Ltd. All Rights Reserved.                   *
 *                                                                                  *
 *    This program is free software; you can redistribute it and/or modify          *
 *    it under the terms of the GNU General Public License, version 2               *
 *    as published by the Free Software Foundation.                                 *
 *                                                                                  *
 *    This program is distributed in the hope that it will be useful,               *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of                *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                 *
 *    GNU General Public License for more details.                                  *
 *                                                                                  *
 *    You should have received a copy of the GNU General Public License along       *
 *    with this program; if not, write to the Free Software Foundation, Inc.,       *
 *    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.                   *
 *                                                                                  *
 ************************************************************************************/

#include <mynahsa/itcpstream.hpp>

#ifndef WIN32
#include <unistd.h>

#include <sys/types.h>
#include <sys/socket.h>

#else
#include <winsock2.h>
#include <windows.h>
#endif


#if BYTE_ORDER == BIG_ENDIAN
#include <mynahsa/byteswap.hpp>
#endif


using namespace std;

namespace MynahSA {

  //! constructor - takes an input stream
  ITCPStream::ITCPStream(int s) : _sd(s), _open(true) {
  }
  
  ITCPStream::ITCPStream(const ITCPStream& i) : _sd(i._sd), _open(i._open) { 
  }
  
  //! destructor - note: TCP is not freed
  ITCPStream::~ITCPStream() {
  }
  
  int ITCPStream::tcpRead(void* buf, int num) { 
    // wrapper for TCP_read; takes care of zero length read - which is not an error, just a closed stream
    // archive
    
    char* cbuf = reinterpret_cast<char*>(buf);
    
    bool done = false;
    
    int originalNum  = num;  // copy of original number of bytes to read
    
    while (!done) { 
      //int result = read(_sd, cbuf, num);
      int result = recv(_sd, cbuf, num, 0);
      if (result < 0) { 
        throw IStreamReadError("ITCPStream::tcpRead An error occured in stream read");
      } else if (result == 0) { 
        throw IStreamEOF();
      }
      
      // note: we're not handling errno yet.
      
      if (result < num) { 
        // handle a short read; this occurs when a write spans multiple TCP records
        cbuf += result;
        num -= result;
      } else { 
        num -= result;
        // num should be zero here
        done = true;
      }
    }
    // assertion check - num should be zero
    
    return originalNum;
  }
  

#if BYTE_ORDER == LITTLE_ENDIAN
#define MAKE_TCP_READER( t ) \
IStreamBase& ITCPStream::operator>>( t & v ) { \
  if (tcpRead( &v, sizeof( t ) ) != sizeof ( t )) { \
    _open = false; \
    throw IStreamReadError("ITCPStream::operator>>( " #t " ) - short read"); \
  } \
  return *this; \
}

#else

#define MAKE_TCP_READER( t ) \
IStreamBase& ITCPStream::operator>>( t & v ) { \
  if (tcpRead( &v, sizeof( t ) ) != sizeof ( t )) { \
    _open = false; \
    throw IStreamReadError("ITCPStream::operator>>( " #t " ) - short read"); \
  } \
  byteswap( v ); \
  return *this; \
}

#endif


  MAKE_TCP_READER( char );
  MAKE_TCP_READER( unsigned char );
  MAKE_TCP_READER( short );
  MAKE_TCP_READER( unsigned short );
  MAKE_TCP_READER( int );
  MAKE_TCP_READER( unsigned int );
  MAKE_TCP_READER( long long );
  MAKE_TCP_READER( unsigned long long );
  MAKE_TCP_READER( double );
  MAKE_TCP_READER( float );
  
#undef MAKE_TCP_READER
  
// this is a code workaround for architectures that use 4 byte bool specifications.  This clamps it onto a 1 
// byte specification which is like x86_64.  After all there is no need to spend more that 1 bit on a bool anyway!
IStreamBase& ITCPStream::operator>>(bool & b ) { 
  char c;
  (*this) >> c;
  b = (c ? true : false); // workaround valgrind hit
  return *this;
}
  
  //! input streaming of std strings
  IStreamBase& ITCPStream::operator>>(string& s) {
    int size=0;
    char* buffer=0;
 
    // recover the size paramter 
    (*this) >> size;
 
    // allocate temporary buffer
    buffer = new char[size+1];
    if (!buffer) { 
      throw IStreamReadError("Buffer memory-allocation failed in std::string reconstruction");
    }
    
    // receive the buffer
    if (tcpRead(buffer,size) != size) { 
      delete []buffer;
      _open = false;
      throw IStreamReadError("Short read in ITCPStream on std::string reconstruction");
    }
  
    // null terminate the string - this is because std::string will read the buffer as a c string
    buffer[size] = '\0';
  
    // copy buffer to string
    s = string(buffer);
    
    // delete buffer
    delete []buffer;
    
    return *this;
  }
  
  void ITCPStream::get(char& c) {
    int result = tcpRead(&c, sizeof(char) );
    if (result != sizeof(char)) { 
      throw IStreamReadError("ITCPStream::get - short read");
    }
  }
};
