/************************************************************************************
 *    This file is part of the MynahSA streaming and archiving toolkit              *
 *    Copyright (C) 2006 Mynah-Software Ltd. All Rights Reserved.                   *
 *                                                                                  *
 *    This program is free software; you can redistribute it and/or modify          *
 *    it under the terms of the GNU General Public License, version 2               *
 *    as published by the Free Software Foundation.                                 *
 *                                                                                  *
 *    This program is distributed in the hope that it will be useful,               *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of                *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                 *
 *    GNU General Public License for more details.                                  *
 *                                                                                  *
 *    You should have received a copy of the GNU General Public License along       *
 *    with this program; if not, write to the Free Software Foundation, Inc.,       *
 *    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.                   *
 *                                                                                  *
 ************************************************************************************/


#include <mynahsa/mkcert.hpp>
#include <mynahsa/sslinit.hpp>

#include <mynahsa/exceptionbase.hpp>

#include <iostream>

using namespace std;

namespace MynahSA { 

  static void callback(int p, int n, void *arg) {
#ifdef DEBUG
    char c='B';
  
    if (p == 0) c='.';
    if (p == 1) c='+';
    if (p == 2) c='*';
    if (p == 3) c='\n';
    cerr.put(c);
#endif
  }
  
  int mkcert(X509 **x509p, EVP_PKEY **pkeyp, int bits, int serial, int days) {
    X509 *x;
    EVP_PKEY *pk = 0;
    RSA *rsa;
    X509_NAME *name=NULL;
  
    if (x509p == 0 || pkeyp == 0) { 
      // this is an error condition; both pointers must be provided
      return 0;
    }
  
    //! if not already done
    sslInit();
    
    if (*pkeyp == NULL)     {
      if ((pk=EVP_PKEY_new()) == NULL) 	{
        throw ExceptionBase("mkcert: Failed to create new private key");
      }
    } else  {
      pk= *pkeyp;
    }
  
    if (*x509p == NULL) {
      if ((x=X509_new()) == NULL) {
        if (*pkeyp != pk) { 
          EVP_PKEY_free(pk);
        }
        throw ExceptionBase("mkcert: Failed to create X509 structure");
      }
    } else { 
      x= *x509p;
    }
    
    rsa=RSA_generate_key(bits,RSA_F4,callback,NULL);
    if (!EVP_PKEY_assign_RSA(pk,rsa)) {
      if (*pkeyp != pk) { 
        EVP_PKEY_free(pk);
      }
      
      if (x != *x509p) { 
        X509_free(x);
      }
        
      throw ExceptionBase("mkcert: Failed to assign RSA key");
    }
    rsa=NULL;
  
    X509_set_version(x,2);
    ASN1_INTEGER_set(X509_get_serialNumber(x),serial);
    X509_gmtime_adj(X509_get_notBefore(x),0);
    X509_gmtime_adj(X509_get_notAfter(x),(long)60*60*24*days);
    X509_set_pubkey(x,pk);
  
    name=X509_get_subject_name(x);
  
    /* This function creates and adds the entry, working out the
    * correct string type and performing checks on its length.
    * Normally we'd check the return value for errors...
    */
    X509_NAME_add_entry_by_txt(name,"C",
            MBSTRING_ASC, (unsigned char*) "UK", -1, -1, 0);
  
    X509_NAME_add_entry_by_txt(name,"CN",
            MBSTRING_ASC, (unsigned char*) "localhost", -1, -1, 0);
  
    /* Its self signed so set the issuer name to be the same as the
    * subject.
    */
    X509_set_issuer_name(x,name);
  
    /* Add various extensions: standard extensions */
    add_ext(x, NID_basic_constraints, "critical,CA:TRUE");
    add_ext(x, NID_key_usage, "critical,keyCertSign,cRLSign");
  
    add_ext(x, NID_subject_key_identifier, "hash");
  
    /* Some Netscape specific extensions */
    add_ext(x, NID_netscape_cert_type, "sslCA");
  
    add_ext(x, NID_netscape_comment, "example comment extension");
  
  
#ifdef CUSTOM_EXT
    /* Maybe even add our own extension based on existing */
    {
      int nid;
      nid = OBJ_create("1.2.3.4", "MyAlias", "My Test Alias Extension");
      X509V3_EXT_add_alias(nid, NID_netscape_comment);
      add_ext(x, nid, "example comment alias");
    }
#endif
    
    if (!X509_sign(x,pk,EVP_md5())) {
      if (*pkeyp != pk) { 
        EVP_PKEY_free(pk);
      }
      
      if (x != *x509p) { 
        X509_free(x);
      }
      throw ExceptionBase("mkcert: Failed to sign key");
    }
    
    // assign pointers for return
    *x509p=x;
    *pkeyp=pk;
    return(1);
  }
  
  /** Add extension using V3 code: we can set the config file as NULL
  *  because we wont reference any other sections.
  */
  int add_ext(X509 *cert, int nid, char *value) {
    X509_EXTENSION *ex;
    X509V3_CTX ctx;
    /* This sets the 'context' of the extensions. */
    /* No configuration database */
    X509V3_set_ctx_nodb(&ctx);
    /* Issuer and subject certs: both the target since it is self signed,
    * no request and no CRL
    */
    X509V3_set_ctx(&ctx, cert, cert, NULL, NULL, 0);
    ex = X509V3_EXT_conf_nid(NULL, &ctx, nid, value);
    if (!ex)
      return 0;
  
    X509_add_ext(cert,ex,-1);
    X509_EXTENSION_free(ex);
    return 1;
  }
};
