/************************************************************************************
 *    This file is part of the MynahSA streaming and archiving toolkit              *
 *    Copyright (C) 2006 Mynah-Software Ltd. All Rights Reserved.                   *
 *                                                                                  *
 *    This program is free software; you can redistribute it and/or modify          *
 *    it under the terms of the GNU General Public License, version 2               *
 *    as published by the Free Software Foundation.                                 *
 *                                                                                  *
 *    This program is distributed in the hope that it will be useful,               *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of                *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                 *
 *    GNU General Public License for more details.                                  *
 *                                                                                  *
 *    You should have received a copy of the GNU General Public License along       *
 *    with this program; if not, write to the Free Software Foundation, Inc.,       *
 *    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.                   *
 *                                                                                  *
 ************************************************************************************/

#include <mynahsa/obinarystream.hpp>


#if BYTE_ORDER == BIG_ENDIAN
#include <mynahsa/byteswap.hpp>
#endif

using namespace std;

namespace MynahSA { 

  OBinaryStream::OBinaryStream(ostream& os) : _os(os) { 
  }
  
  OBinaryStream::OBinaryStream(const OBinaryStream& os) : _os(os._os) { 
  }
  
  OBinaryStream::~OBinaryStream() {
  }
  
  OStreamBase& OBinaryStream::operator<<(const char c) {
    _os.put(c);
    return *this;
  }
  
  OStreamBase& OBinaryStream::operator<<(const unsigned char c) {
    _os.put(c);
    return *this;
  }
  
#if BYTE_ORDER == LITTLE_ENDIAN
#define MAKE_BIN_SAVER( x ) OStreamBase& OBinaryStream::operator<<(x c) {\
  char* cp = (char*) &(c);\
  for (unsigned int i=0; i<sizeof(c); i++) { \
    _os.put(cp[i]);\
  }\
  return *this;\
}
#else
#define MAKE_BIN_SAVER( x ) OStreamBase& OBinaryStream::operator<<(x c) {\
  byteswap(c); \
  char* cp = (char*) &(c);\
  for (unsigned int i=0; i<sizeof(c); i++) { \
    _os.put(cp[i]);\
  }\
  return *this;\
}
#endif
  
  MAKE_BIN_SAVER( const short );
  MAKE_BIN_SAVER( const unsigned short );
  MAKE_BIN_SAVER( const int );
  MAKE_BIN_SAVER( const unsigned int );
  MAKE_BIN_SAVER( const long long );
  MAKE_BIN_SAVER( const unsigned long long );
  MAKE_BIN_SAVER( const double );
  MAKE_BIN_SAVER( const float );
  
#undef MAKE_BIN_SAVER

  // This is a workaround for some architectures that keep bools as 4 byte members.  X86_64 specifies a 1 byte bool,
  // and we will keep it this way on the stream.
  OStreamBase& OBinaryStream::operator<<(bool b) { 
    char c = b;
    (*this) << c;
    return *this;
  }


};

