/************************************************************************************
 *    This file is part of the MynahSA streaming and archiving toolkit              *
 *    Copyright (C) 2006 Mynah-Software Ltd. All Rights Reserved.                   *
 *                                                                                  *
 *    This program is free software; you can redistribute it and/or modify          *
 *    it under the terms of the GNU General Public License, version 2               *
 *    as published by the Free Software Foundation.                                 *
 *                                                                                  *
 *    This program is distributed in the hope that it will be useful,               *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of                *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                 *
 *    GNU General Public License for more details.                                  *
 *                                                                                  *
 *    You should have received a copy of the GNU General Public License along       *
 *    with this program; if not, write to the Free Software Foundation, Inc.,       *
 *    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.                   *
 *                                                                                  *
 ************************************************************************************/

#include <mynahsa/oblowfishstream.hpp>    
    
#ifndef WIN32
#include <strings.h>
#else
#define bzero(thing, sz) memset((thing), 0, (sz));
#endif

#if BYTE_ORDER == BIG_ENDIAN
#include <mynahsa/byteswap.hpp>
#endif



#include <iostream>


using namespace std;
    
    
    
namespace MynahSA { 
  OBlowfishStream::OBlowfishStream(std::ostream& os, const std::string& key) : _os(os), _buffPtr(0), _open(true) {
    bzero(_buffer, 8);
    bzero(_outBuffer, 8);
    bzero(_ivec, 8);
    bzero(&_bfKey, sizeof(BF_KEY)); // zero the blowfish key
  
    // initialize the key
    BF_set_key(&_bfKey, key.size(), (const unsigned char*) key.c_str());
  
  }
  
  OBlowfishStream::OBlowfishStream(const OBlowfishStream& os) : _os(os._os), 
                                                                _buffPtr(os._buffPtr), 
                                                                _open(os._open) {
    memcpy(_buffer, os._buffer, 8);
    memcpy(_outBuffer, os._outBuffer, 8);
    memcpy(_ivec, os._ivec, 8);
    memcpy(&(_bfKey), &(os._bfKey), sizeof(BF_KEY));
  }
  
  OBlowfishStream::~OBlowfishStream() {
    // flush stream prior to closing
    close();
  }
  
  void OBlowfishStream::flush() { 
    if (_buffPtr == 0) { 
      // nothing to do.
      return;
    } 
    
    while (_buffPtr < 8) { 
      _buffer[_buffPtr++] = 0;  // insert a pad byte
    }
    
    // encrypt the data
    BF_cbc_encrypt(_buffer, 
                   _outBuffer, 
                   8, 
                   &_bfKey,
                   _ivec,
                   BF_ENCRYPT);
    
    for (unsigned int i=0; i<8; i++) { 
      _os.put( (char) _outBuffer[i]);
    }
    
    _buffPtr = 0;
  }
  
  void OBlowfishStream::close() { 
    flush();
    _open = false;
  }
  
  
  void OBlowfishStream::bfWriteChar(unsigned char value) {
  
    if (!_open) { 
      throw OStreamWriteError("OBlowfishStream::bfWrite - attempt to write to a closed stream");
    }
  
    _buffer[_buffPtr++] = value;
    if (_buffPtr >= 8) { 
      // transmit the data and reset the buffer state
      flush();
    }
  }
  
  
#if BYTE_ORDER == LITTLE_ENDIAN
// intel codepath
#define MAKE_BF_WRITER( x ) \
  OStreamBase& OBlowfishStream::operator<<( x j) { \
    const unsigned char* ucharp = reinterpret_cast<const unsigned char*>(&j); \
    for (unsigned int i=0; i<sizeof( x ); i++) { \
      bfWriteChar(ucharp[i]); \
    } \
    return *this; \
  }
#else
// ppc / sparc / mips codepath
#define MAKE_BF_WRITER( x ) \
  OStreamBase& OBlowfishStream::operator<<( x j) { \
    byteswap(j); \
    const unsigned char* ucharp = reinterpret_cast<const unsigned char*>(&j); \
    for (unsigned int i=0; i<sizeof( x ); i++) { \
      bfWriteChar(ucharp[i]); \
    } \
    return *this; \
  }
#endif
  
  MAKE_BF_WRITER(const char);    
  MAKE_BF_WRITER(const unsigned char);
  MAKE_BF_WRITER(const short);
  MAKE_BF_WRITER(const unsigned short);
  MAKE_BF_WRITER(const int);
  MAKE_BF_WRITER(const unsigned int);
  MAKE_BF_WRITER(const long long);
  MAKE_BF_WRITER(const unsigned long long);
  MAKE_BF_WRITER(const double);
  MAKE_BF_WRITER(const float);   

#undef MAKE_BF_WRITER

// This is a workaround for some architectures that keep bools as 4 byte members.  X86_64 specifies a 1 byte bool,
  // and we will keep it this way on the stream.
  OStreamBase& OBlowfishStream::operator<<(bool b) { 
    char c = b;
    (*this) << c;
    return *this;
  }


};  // closes namespace 
