/************************************************************************************
 *    This file is part of the MynahSA streaming and archiving toolkit              *
 *    Copyright (C) 2006 Mynah-Software Ltd. All Rights Reserved.                   *
 *                                                                                  *
 *    This program is free software; you can redistribute it and/or modify          *
 *    it under the terms of the GNU General Public License, version 2               *
 *    as published by the Free Software Foundation.                                 *
 *                                                                                  *
 *    This program is distributed in the hope that it will be useful,               *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of                *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                 *
 *    GNU General Public License for more details.                                  *
 *                                                                                  *
 *    You should have received a copy of the GNU General Public License along       *
 *    with this program; if not, write to the Free Software Foundation, Inc.,       *
 *    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.                   *
 *                                                                                  *
 ************************************************************************************/

#include <mynahsa/osslstream.hpp>

#define BUFFER_TRANSMIT_SIZE 16384

#ifndef WIN32
#include <strings.h>
#else
#define bzero(thing, sz) memset((thing), 0, (sz));
#endif



#if BYTE_ORDER == BIG_ENDIAN
#include <mynahsa/byteswap.hpp>
#endif


using namespace std;

namespace MynahSA { 

  OSSLStream::OSSLStream(SSL* s) : _ssl(s), _buffPtr(0) {
    _buffer = new char[BUFFER_TRANSMIT_SIZE];
    bzero(_buffer, BUFFER_TRANSMIT_SIZE);  // clear buffer
  }
  
  OSSLStream::OSSLStream(const OSSLStream& s) : _ssl(s._ssl), _buffPtr(s._buffPtr) { 
    _buffer = new char[BUFFER_TRANSMIT_SIZE];
    memcpy(_buffer, s._buffer, BUFFER_TRANSMIT_SIZE); // copy buffer
  }
  
  OSSLStream::~OSSLStream() {
    delete []_buffer;
    _buffer = 0;
  }
  
  
  int OSSLStream::sslWrite(const void* data, unsigned int size) {
    // recast pointer to const char
    const char* cdata = reinterpret_cast<const char*>(data);
    
    // now: we'll copy the data onto the buffer, treating the buffer as a ring.  If we overfill the end of the buffer
    // we will induce an SSL_write call to flush the buffer.
    for (unsigned int i=0; i<size; i++) { 
      _buffer[_buffPtr++] = cdata[i];
      if (_buffPtr == BUFFER_TRANSMIT_SIZE) { 
        int result = SSL_write(_ssl, _buffer, BUFFER_TRANSMIT_SIZE);
        if (result != BUFFER_TRANSMIT_SIZE) { // an error?
          return 0;  // buffer is hosed!
          //throw OStreamWriteError("OSSLStream::sslWrite - failure in flushing buffer during write operation.");
        }
        _buffPtr = 0;  // reset buffer pointer
      }
    }
    return size;
  }
  
  void OSSLStream::flush() { 
    // transmit any data that sits in the buffer and return
    int result = SSL_write(_ssl, _buffer, _buffPtr);
    if (result != _buffPtr) {
      throw OStreamWriteError("OSSLStream::flush() - failure in flushing buffer!");
    }
    _buffPtr = 0;  // reset the buffer pointer
  }
  
  // Now instantiate outstream overrides
  // NOTE: We'll add machine independant serialization at a later date

#if BYTE_ORDER == LITTLE_ENDIAN

#define MAKE_SSL_SAVER( t ) OStreamBase& OSSLStream::operator<<( t v ) { \
  if (sslWrite(&v, sizeof(v)) != sizeof(v)) {\
    throw OStreamWriteError("OSSLStream::operator<<(" #t ") - Failed to write.");\
  }\
  return *this;\
}

#else

#define MAKE_SSL_SAVER( t ) OStreamBase& OSSLStream::operator<<( t v ) { \
  byteswap( v ); \
  if (sslWrite(&v, sizeof(v)) != sizeof(v)) {\
    throw OStreamWriteError("OSSLStream::operator<<(" #t ") - Failed to write.");\
  }\
  return *this;\
}


#endif

  MAKE_SSL_SAVER( const char );
  MAKE_SSL_SAVER( const unsigned char );
  MAKE_SSL_SAVER( const short );
  MAKE_SSL_SAVER( const unsigned short );
  MAKE_SSL_SAVER( const int );
  MAKE_SSL_SAVER( const unsigned int );
  MAKE_SSL_SAVER( const long long );
  MAKE_SSL_SAVER( const unsigned long long );
  MAKE_SSL_SAVER( const double );    
  MAKE_SSL_SAVER( const float );
  
#undef MAKE_SSL_SAVER
  
  // This is a workaround for some architectures that keep bools as 4 byte members.  X86_64 specifies a 1 byte bool,
  // and we will keep it this way on the stream.
  OStreamBase& OSSLStream::operator<<(bool b) { 
    char c = b;
    (*this) << c;
    return *this;
  }
  
  
  OStreamBase& OSSLStream::operator<<(const string& s) {
    (*this) << ((unsigned int) s.size());
    if (s.size() == 0) { 
      // early terminate zero length strings
      return *this;
    }
    if (sslWrite( s.c_str(), s.size()) != s.size()) { 
      throw OStreamWriteError("Short write in streaming a std::string");
    }
  
    return *this;
  }
  
#undef BUFFER_TRANSMIT_SIZE
};
