/************************************************************************************
 *    This file is part of the MynahSA streaming and archiving toolkit              *
 *    Copyright (C) 2006 Mynah-Software Ltd. All Rights Reserved.                   *
 *                                                                                  *
 *    This program is free software; you can redistribute it and/or modify          *
 *    it under the terms of the GNU General Public License, version 2               *
 *    as published by the Free Software Foundation.                                 *
 *                                                                                  *
 *    This program is distributed in the hope that it will be useful,               *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of                *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                 *
 *    GNU General Public License for more details.                                  *
 *                                                                                  *
 *    You should have received a copy of the GNU General Public License along       *
 *    with this program; if not, write to the Free Software Foundation, Inc.,       *
 *    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.                   *
 *                                                                                  *
 ************************************************************************************/

#include <mynahsa/otcpstream.hpp>

#ifndef WIN32
#include <unistd.h>
#include <strings.h>

#include <sys/types.h>
#include <sys/socket.h>

#else
#include <windows.h>
#include <winsock2.h>
#define bzero(thing, sz) memset((thing), 0, (sz));
#endif

#if BYTE_ORDER == BIG_ENDIAN
#include <mynahsa/byteswap.hpp>
#endif


// we set the buffer size to 1400 - this is an initial guess that should get us close to the TCP MTU
#define BUFFER_TRANSMIT_SIZE 1400

using namespace std;

namespace MynahSA { 

  OTCPStream::OTCPStream(int s) : _sd(s), _buffPtr(0) {
    _buffer = new char[BUFFER_TRANSMIT_SIZE];
    bzero(_buffer, BUFFER_TRANSMIT_SIZE);  // clear buffer
  }
  
  OTCPStream::OTCPStream(const OTCPStream& s) : _sd(s._sd), _buffPtr(s._buffPtr) { 
    _buffer = new char[BUFFER_TRANSMIT_SIZE];
    memcpy(_buffer, s._buffer, BUFFER_TRANSMIT_SIZE); // copy buffer
  }
  
  OTCPStream::~OTCPStream() {
    delete []_buffer;
    _buffer = 0;
  }
  
  
  int OTCPStream::tcpWrite(const void* data, unsigned int size) {
    // recast pointer to const char
    const char* cdata = reinterpret_cast<const char*>(data);
    
    // now: we'll copy the data onto the buffer, treating the buffer as a ring.  If we overfill the end of the buffer
    // we will induce an TCP_write call to flush the buffer.
    for (unsigned int i=0; i<size; i++) { 
      _buffer[_buffPtr++] = cdata[i];
      if (_buffPtr == BUFFER_TRANSMIT_SIZE) { 
        //int result = write(_sd, _buffer, BUFFER_TRANSMIT_SIZE);
        int result = send(_sd, _buffer, BUFFER_TRANSMIT_SIZE, 0);
        if (result != BUFFER_TRANSMIT_SIZE) { // an error?
          return 0;  // buffer is hosed!
          //throw OStreamWriteError("OTCPStream::sslWrite - failure in flushing buffer during write operation.");
        }
        _buffPtr = 0;  // reset buffer pointer
      }
    }
    return size;
  }
  
  void OTCPStream::flush() { 
    // transmit any data that sits in the buffer and return
    //int result = write(_sd, _buffer, _buffPtr);
    int result = send(_sd, _buffer, _buffPtr, 0);
    if (result != _buffPtr) {
      throw OStreamWriteError("OTCPStream::flush() - failure in flushing buffer!");
    }
    _buffPtr = 0;  // reset the buffer pointer
  }
  
  // Now instantiate outstream overrides
  // NOTE: We'll add machine independant serialization at a later date

#if BYTE_ORDER == LITTLE_ENDIAN

#define MAKE_TCP_SAVER( t ) OStreamBase& OTCPStream::operator<<( t v ) { \
  if (tcpWrite(&v, sizeof(v)) != sizeof(v)) {\
    throw OStreamWriteError("OTCPStream::operator<<(" #t ") - Failed to write.");\
  }\
  return *this;\
}

#else

#define MAKE_TCP_SAVER( t ) OStreamBase& OTCPStream::operator<<( t v ) { \
  byteswap( v ); \
  if (tcpWrite(&v, sizeof(v)) != sizeof(v)) {\
    throw OStreamWriteError("OTCPStream::operator<<(" #t ") - Failed to write.");\
  }\
  return *this;\
}


#endif


  MAKE_TCP_SAVER( const char );
  MAKE_TCP_SAVER( const unsigned char );
  MAKE_TCP_SAVER( const short );
  MAKE_TCP_SAVER( const unsigned short );
  MAKE_TCP_SAVER( const int );
  MAKE_TCP_SAVER( const unsigned int );
  MAKE_TCP_SAVER( const long long );
  MAKE_TCP_SAVER( const unsigned long long );
  MAKE_TCP_SAVER( const double );    
  MAKE_TCP_SAVER( const float );
  
#undef MAKE_TCP_SAVER
  
  // This is a workaround for some architectures that keep bools as 4 byte members.  X86_64 specifies a 1 byte bool,
  // and we will keep it this way on the stream.
  OStreamBase& OTCPStream::operator<<(bool b) { 
    char c = b;
    (*this) << c;
    return *this;
  }
  
  
  OStreamBase& OTCPStream::operator<<(const string& s) {
    (*this) << ((unsigned int) s.size());
    if (tcpWrite( s.c_str(), s.size()) != s.size()) { 
      throw OStreamWriteError("Short write in streaming a std::string");
    }
  
    return *this;
  }
  
#undef BUFFER_TRANSMIT_SIZE
};
