/************************************************************************************
 *    This file is part of the MynahSA streaming and archiving toolkit              *
 *    Copyright (C) 2006 Mynah-Software Ltd. All Rights Reserved.                   *
 *                                                                                  *
 *    This program is free software; you can redistribute it and/or modify          *
 *    it under the terms of the GNU General Public License, version 2               *
 *    as published by the Free Software Foundation.                                 *
 *                                                                                  *
 *    This program is distributed in the hope that it will be useful,               *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of                *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                 *
 *    GNU General Public License for more details.                                  *
 *                                                                                  *
 *    You should have received a copy of the GNU General Public License along       *
 *    with this program; if not, write to the Free Software Foundation, Inc.,       *
 *    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.                   *
 *                                                                                  *
 ************************************************************************************/

#include <mynahsa/sslconnectioncertverifier.hpp>

#include <iostream>
#include <string>
#ifndef WIN32
#include <sys/types.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <resolv.h>
#include <unistd.h>
#include <netdb.h>
#else
#include <winsock2.h>
#include <ws2tcpip.h>
#include <windows.h>
#endif

using namespace std;

namespace MynahSA { 

  SSLConnectionCertVerifier::SSLConnectionCertVerifier() {
  }
  
  SSLConnectionCertVerifier::~SSLConnectionCertVerifier() { 
  }
  
  bool SSLConnectionCertVerifier::operator()(SSL* ssl) const { 
    // assume certificate provided by peer is rejected
    bool result = false;
#ifdef DEBUG
    cerr << "SSLConnectionCertVerifier:Beginning verification process" << endl;
#endif
  
    
    if (SSL_get_verify_result(ssl) != X509_V_OK) {
#ifdef DEBUG
      cerr << "SSLConnectionCertVerifier: SSL_get_verify_result failed; returning " << endl;
#endif
      return false;
    }
    
    X509* peer;
    peer = SSL_get_peer_certificate(ssl); 
    if (peer == 0) { 
#ifdef DEBUG
      cerr << "NO client certificate - returning " << endl;
#endif
      return false;
    }
#ifdef DEBUG
      else { 
      cerr << "Got a certificate" << endl;
    }
#endif
    
    // using example code from Rescorla
    char peerCN[1024];
    if (X509_NAME_get_text_by_NID(X509_get_subject_name(peer), NID_commonName, peerCN, sizeof(peerCN) ) != -1) { 
#ifdef DEBUG
      std::cerr << "SSLConnectionCertVerifier: peer certificate CN is: " << peerCN << std::endl;
#endif
    
      // fetch file descriptor (socket connection)
      int s = SSL_get_fd(ssl);
      
      // get the peer name of the attached machine
      struct sockaddr_in addr;
  
#ifdef WIN32
      int len = sizeof(addr);
#else
      socklen_t len = sizeof(addr);
#endif
  
      int xresult = getpeername(s, (struct sockaddr*) &addr, &len);
      if (xresult == 0) { // success in finding peer name
        // convert the ipaddress to numbers and dots form
        char buffer[32];
#ifndef WIN32
        inet_ntop(AF_INET, &addr.sin_addr.s_addr, buffer, sizeof(buffer));
#else
        // This should be thread safe - if the msdn description is correct
        struct in_addr ia;
        ia.S_un.S_addr = addr.sin_addr.s_addr;
        const char* sp = inet_ntoa(ia);
        int len = strlen(sp);
        len += 1; // copy the \0
        for (unsigned int i =0; i<len && i < sizeof(buffer); i++) { 
          buffer[i] = sp[i];
        }
#endif
        
#ifdef DEBUG
        std::cerr << "SSLConnectionCertVerifier: Connection from: " << buffer << "  ("
          << ntohs(addr.sin_port) << ")\n";
#endif
          
        // great now lookup the hostname by ipaddress, with the name, we'll compare it against the certificate name
        char hostname[1024];
        int niResult = getnameinfo((struct sockaddr*) &addr, sizeof(addr),
                                  hostname, sizeof(hostname),
                                  0, 0,
                                  NI_NAMEREQD);
        
        if (niResult == 0) { 
#ifdef DEBUG
          cerr << "SSLConnectionCertVerifier: Host name is: " << hostname << endl;
#endif
          if (string(peerCN) == string(hostname)) { 
            // match - accept this client
#ifdef DEBUG
            cerr << "SSLConnectionCertVerifier: Accepting host - FQDN (by DNS) matches peer certificate CN" << endl;
#endif
            result = true;
          }
        }  // error in finding FQDN
      }
    }
    X509_free(peer);
    return result;
  }
};

