/************************************************************************************
 *    This file is part of the MynahSA streaming and archiving toolkit              *
 *    Copyright (C) 2006 Mynah-Software Ltd. All Rights Reserved.                   *
 *                                                                                  *
 *    This program is free software; you can redistribute it and/or modify          *
 *    it under the terms of the GNU General Public License, version 2               *
 *    as published by the Free Software Foundation.                                 *
 *                                                                                  *
 *    This program is distributed in the hope that it will be useful,               *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of                *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                 *
 *    GNU General Public License for more details.                                  *
 *                                                                                  *
 *    You should have received a copy of the GNU General Public License along       *
 *    with this program; if not, write to the Free Software Foundation, Inc.,       *
 *    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.                   *
 *                                                                                  *
 ************************************************************************************/


// History:
// BMS20060307 - added thread code support from OpenSSL sample code



#include <mynahsa/sslinit.hpp>

#include <openssl/ssl.h>
#include <openssl/rand.h>

#include <iostream>

using namespace std;

namespace MynahSA { 

  static unsigned int __ssl_initialized = 0;
  
  static const char rnd_seed[] = "string to make the random number generator think it has entropy";
  
  
#ifdef WIN32
  
#include <windows.h>
  
  static HANDLE *lock_cs;
  
  void thread_cleanup(void) {
    int i;
    
    CRYPTO_set_locking_callback(NULL);
    for (i=0; i<CRYPTO_num_locks(); i++)
      CloseHandle(lock_cs[i]);
    OPENSSL_free(lock_cs);
  }
  
  void win32_locking_callback(int mode, int type, const char *file, int line) {
    if (mode & CRYPTO_LOCK)   {
      WaitForSingleObject(lock_cs[type],INFINITE);
    } else {
      ReleaseMutex(lock_cs[type]);
    }
  }
  
  void thread_setup(void) {
    int i;
  
    lock_cs= (HANDLE*) OPENSSL_malloc(CRYPTO_num_locks() * sizeof(HANDLE));
    for (i=0; i<CRYPTO_num_locks(); i++) {
      lock_cs[i]=CreateMutex(NULL,FALSE,NULL);
    }
    
    CRYPTO_set_locking_callback((void (*)(int,int,const char *,int))win32_locking_callback);
    /* id callback defined */
  }
  
  
#else
  
  // posix implementation
  
#include <pthread.h>
  
  static pthread_mutex_t *lock_cs;
  static long *lock_count;
  
  
  void thread_cleanup(void) {
    int i;
    
    CRYPTO_set_locking_callback(NULL);
    //fprintf(stderr,"cleanup\n");
    for (i=0; i<CRYPTO_num_locks(); i++) {
      pthread_mutex_destroy(&(lock_cs[i]));
      //fprintf(stderr,"%8ld:%s\n",lock_count[i],
      //    CRYPTO_get_lock_name(i));
    }
    OPENSSL_free(lock_cs);
    OPENSSL_free(lock_count);
    
    //fprintf(stderr,"done cleanup\n");
  }
  void pthreads_locking_callback(int mode, int type, char *file,
              int line) {
#ifdef DEBUG_OPENSSL_THREAD
    fprintf(stderr,"thread=%4d mode=%s lock=%s %s:%d\n",
      CRYPTO_thread_id(),
      (mode&CRYPTO_LOCK)?"l":"u",
      (type&CRYPTO_READ)?"r":"w",file,line);
#endif
  /*
    if (CRYPTO_LOCK_SSL_CERT == type)
    fprintf(stderr,"(t,m,f,l) %ld %d %s %d\n",
    CRYPTO_thread_id(),
    mode,file,line);
  */
    if (mode & CRYPTO_LOCK) {
      pthread_mutex_lock(&(lock_cs[type]));
      lock_count[type]++;
    } else {
      pthread_mutex_unlock(&(lock_cs[type]));
    }
  }
  
  unsigned long pthreads_thread_id(void) {
    unsigned long ret;
    
    ret=(unsigned long)pthread_self();
    return(ret);
  }
  
  void thread_setup(void)	{
    int i;
    
    lock_cs = (pthread_mutex_t*) OPENSSL_malloc(CRYPTO_num_locks() * sizeof(pthread_mutex_t));
    lock_count = (long*) OPENSSL_malloc(CRYPTO_num_locks() * sizeof(long));
    for (i=0; i<CRYPTO_num_locks(); i++) {
      lock_count[i]=0;
      pthread_mutex_init(&(lock_cs[i]),NULL);
    }
  
    CRYPTO_set_id_callback((unsigned long (*)())pthreads_thread_id);
    CRYPTO_set_locking_callback((void (*)(int,int,const char*, int))pthreads_locking_callback);
  }
  
  
#endif /* PTHREADS */
  
  
  void sslInit() {
    if (__ssl_initialized != 1) { 
#ifdef DEBUG
      cerr << "Initializing SSL library" << endl;
  
      CRYPTO_malloc_debug_init();
      CRYPTO_dbg_set_options(V_CRYPTO_MDEBUG_ALL);
      CRYPTO_mem_ctrl(CRYPTO_MEM_CHECK_ON);
#endif
    
  
      // Load algorithms and error strings.
      SSL_load_error_strings();
  
      // initialize SSL library
      SSL_library_init();
  
      // initialize PRNG
  
      // from rsa example
      RAND_seed(rnd_seed, sizeof rnd_seed);
      
      // initialize thread support
      thread_setup();
  
      // set initialized flag
      __ssl_initialized = 1;
    }
  }
};
