/************************************************************************************
 *    This file is part of the MynahSA streaming and archiving toolkit              *
 *    Copyright (C) 2006 Mynah-Software Ltd. All Rights Reserved.                   *
 *                                                                                  *
 *    This program is free software; you can redistribute it and/or modify          *
 *    it under the terms of the GNU General Public License, version 2               *
 *    as published by the Free Software Foundation.                                 *
 *                                                                                  *
 *    This program is distributed in the hope that it will be useful,               *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of                *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                 *
 *    GNU General Public License for more details.                                  *
 *                                                                                  *
 *    You should have received a copy of the GNU General Public License along       *
 *    with this program; if not, write to the Free Software Foundation, Inc.,       *
 *    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.                   *
 *                                                                                  *
 ************************************************************************************/

#include <mynahsa/tcpconnectionmanager.hpp>

#include <iostream>
#include <stdlib.h>
#include <stdio.h>
#include <time.h>

#include <string.h>

#ifdef WIN32
#include <winsock2.h>
#include <windows.h>
#define bzero(thing,bz) memset((thing),0,(bz))
#else
#include <unistd.h>
#endif


#include <mynahsa/tcparchivestream.hpp>


namespace MynahSA { 

  //! constructor - hand the constructor a request handler object
  TCPConnectionManager::TCPConnectionManager(RequestResponseHandler& handler,
                                              const StreamConstructor& cons) : _handler(handler),
                                                                               _streamConstructor(cons) {
  }

  TCPConnectionManager::~TCPConnectionManager() {
  }

  void TCPConnectionManager::operator()(int fd) { 
#ifdef DEBUG
    std::cerr << "****************************** ConnectionManager::operator() ******************************" << std::endl;
#endif
    
    // Instantiate a stream on the TCP connection - use the stream constructor function to perform 
    // necessary registrations
    TCPArchiveStream stream(fd);  // construct stream on fd
    _streamConstructor(stream);   // initialize all constructors on stream
    
    try {
      // Now: while the stream is open
      while (stream.is_open()) {
  
        SHARED_PTR<IoBase> req;    
        stream >> req;                  // receive a request
        stream.clearUPR();              // clear Unique Pointer References
        
        SHARED_PTR<IoBase> rsp;  // response
        rsp = _handler(req);
  
        stream << rsp;         // compute a response, and push it down the stream
        stream.flush();        // flush stream 
        stream.clearUPR();     // clear unique pointer references
        
      }
    } catch (const IStreamEOF&) { 
      // this occurs when the stream is closed
    } catch(const IStreamReadError& r) {
      std::cerr << "ITCPStreamReadError: " << r.what() << std::endl;
    } catch(const OStreamWriteError& w) { 
      std::cerr << "OTCPStreamWriteError: " << w.what() << std::endl;
    } catch(const std::logic_error& le) { 
      std::cerr << "logic_error: " << le.what() << std::endl;
    }
          
    // close the stream
    stream.close();
  
#ifdef WIN32
    closesocket(fd);
#else
    close(fd);
#endif
  
#ifdef DEBUG
    std::cerr << "Connection Closed" << std::endl;
    std::cerr << "---------------------------------------------" << std::endl;
#endif
  }
};

