/************************************************************************************
 *    This file is part of the MynahSA streaming and archiving toolkit              *
 *    Copyright (C) 2006 Mynah-Software Ltd. All Rights Reserved.                   *
 *                                                                                  *
 *    This program is free software; you can redistribute it and/or modify          *
 *    it under the terms of the GNU General Public License, version 2               *
 *    as published by the Free Software Foundation.                                 *
 *                                                                                  *
 *    This program is distributed in the hope that it will be useful,               *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of                *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                 *
 *    GNU General Public License for more details.                                  *
 *                                                                                  *
 *    You should have received a copy of the GNU General Public License along       *
 *    with this program; if not, write to the Free Software Foundation, Inc.,       *
 *    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.                   *
 *                                                                                  *
 ************************************************************************************/

#include <mynahsa/tcprpcclient.hpp>

#include <mynahsa/tcparchivestream.hpp>

#ifndef WIN32
#include <unistd.h>
#include <strings.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <sys/socket.h>
#include <signal.h>
#else
#include <windows.h>
#include <winsock.h>
#define bzero(thing, sz) memset((thing), 0, (sz));
#endif


#include <mynahsa/sainit.hpp>

#include <fcntl.h>

#include <sstream>

#include <iostream>

#include <mynahsa/clientexception.hpp>

using namespace std;

namespace MynahSA {
  
  void TCPRPCClient::init(const StreamConstructor& sc,
                             unsigned int ipaddress,
                             unsigned int port) {
                                                          
                                                          
#ifndef WIN32
    //BMS20051230: Don't know the unix equiv for this
    // note: this is disabling signal SIGPIPE
    signal(SIGPIPE, SIG_IGN);
#endif                                                            
                                                          
     _sd = socket(AF_INET, SOCK_STREAM, 0);
  
    if (_sd < 0) {
      throw ClientException(ClientException::CLIENT_ERROR_CREATE_SOCKET);
    }
  
    int on=1;
    
    // configure socket for reuse.
#ifdef WIN32
    int sockres = setsockopt(_sd, SOL_SOCKET, SO_REUSEADDR, (const char*) &on, sizeof(on));
#else
    int sockres = setsockopt(_sd, SOL_SOCKET, SO_REUSEADDR, &on, sizeof(on));
#endif
  
    if (sockres != 0) {
      throw ClientException(ClientException::CLIENT_ERROR_SOCKET_REUSE);
    }
  
  
    struct sockaddr_in sa;  // socket address structure
    bzero(&sa, sizeof(sa)); // and clear it
    
    sa.sin_family = AF_INET;
    sa.sin_addr.s_addr = ipaddress;
    sa.sin_port = htons(port);
  
    int err = connect(_sd, (struct sockaddr*) &sa, sizeof(sa));
    if (err < 0) {
      // warning: memory lost here on exception
      throw  ClientException(ClientException::CLIENT_ERROR_CONNECT);
    }
    
    // now ssl _ssl is bound; create an TCP stream using _ssl
    _stream = SHARED_PTR<TCPArchiveStream>(new TCPArchiveStream(_sd));                      
    sc(*_stream);  // register things with stream constructor    
    // that's it.  The stream is ready.
  
    _isopen = true;
    
#ifdef DEBUG
    cerr << "Client setup complete" << endl;
#endif                            
                             
  }                             
  
  TCPRPCClient::TCPRPCClient(const StreamConstructor& sc,
                             unsigned int ipaddress,
                             unsigned int port) : _isopen(false),
                                                  _sd(0) { 
    // initialize tcp interface.  No need to force ssl initialization                                                  
    saInit(true,false);                                                  
    init(sc,ipaddress,port);
  }
  
  
  TCPRPCClient::TCPRPCClient(const StreamConstructor& sc,
                             string machineName,
                             unsigned int port) : _isopen(false),
                                                  _sd(0) { 
    // initialize tcp interface.  No need to force ssl initialization                                                  
    saInit(true,false);     
                                                 
                                                                
    // look up the host
    struct hostent* hid;
    hid = gethostbyname(machineName.c_str());
    
    if (hid==0) { 
      throw ClientException(ClientException::CLIENT_NO_ADDRESS);
    }
    
    ostringstream ss;
    ss << ((unsigned int) ((unsigned char) hid->h_addr[0])) << "."
       << ((unsigned int) ((unsigned char) hid->h_addr[1])) << "."
       << ((unsigned int) ((unsigned char) hid->h_addr[2])) << "."
       << ((unsigned int) ((unsigned char) hid->h_addr[3]));

#ifdef WIN32
    unsigned long ina;
    ina = inet_addr(ss.str().c_str());
#else
    in_addr_t ina;
    // cerr << "haddr is: " << ss.str() << endl;
    if (!inet_aton(ss.str().c_str(), (struct in_addr*) &ina)) {
      throw ClientException(ClientException::CLIENT_ADDRESS_CONVERSION_ERROR);
    }
#endif                                                  
                                                     
    init(sc,ina,port);
  }  
  
  TCPRPCClient::~TCPRPCClient() { 
    // close will free allocated datastructures if necessary.
    close();
  }
  
  SHARED_PTR<IoBase> TCPRPCClient::rpc(SHARED_PTR<IoBase> req) { 
  
    // Send object to server
    (*(_stream)) << req;
    _stream->flush(); // flush the stream!
    _stream->clearUPR();  // clear unique pointer references
  
  
    // receive response from server
    SHARED_PTR<IoBase> resp;
    *(_stream) >> resp;
    _stream->clearUPR();  // and clear any received UPRs
    
    // return response to caller
    return resp;  
  }
  
  void TCPRPCClient::close() { 
  
#ifdef WIN32
    closesocket(_sd);
#else
    ::close(_sd);
#endif
  
    _isopen = false;
  };
};
