/************************************************************************************
 *    This file is part of the MynahSA streaming and archiving toolkit              *
 *    Copyright (C) 2006 Mynah-Software Ltd. All Rights Reserved.                   *
 *                                                                                  *
 *    This program is free software; you can redistribute it and/or modify          *
 *    it under the terms of the GNU General Public License, version 2               *
 *    as published by the Free Software Foundation.                                 *
 *                                                                                  *
 *    This program is distributed in the hope that it will be useful,               *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of                *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                 *
 *    GNU General Public License for more details.                                  *
 *                                                                                  *
 *    You should have received a copy of the GNU General Public License along       *
 *    with this program; if not, write to the Free Software Foundation, Inc.,       *
 *    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.                   *
 *                                                                                  *
 ************************************************************************************/

#include <mynahsa/tcpserver.hpp>

#ifndef WIN32
#include <sys/socket.h>
#include <arpa/inet.h>
#include <resolv.h>
#include <unistd.h>
#else
#include <winsock2.h>
#include <windows.h>
#endif

#include <stdio.h>
#include <string.h>
#include <iostream>

#include <mynahsa/thread.hpp>

#include <signal.h>

#include <iostream>

#include <mynahsa/serverexception.hpp>

#include <fcntl.h>

namespace MynahSA { 
  TCPServer::TCPServer(TCPConnectionManager& sobj, 
                       int port) : _serverObject(sobj), 
                                   _port(port) {
    

    // Get the server started.
    bindPort();

#ifndef WIN32
    //BMS20051230: Don't know the unix equiv for this
    // note: this is disabling signal SIGPIPE
    signal(SIGPIPE, SIG_IGN);
#endif
  
  }
  
  TCPServer::~TCPServer() { 
    // close port
#ifndef WIN32
    close(_master);
#else
    closesocket(_master);
#endif
  }
  
  
  struct tcp_thread_hider { 
    tcp_thread_hider(TCPConnectionManager& obj, int j) :_obj(obj), _j(j) { 
    }
    void operator()() {
      _obj(_j);
    }
    TCPConnectionManager& _obj;
    int _j;
  };
        
  
  void TCPServer::checkClients(int wait_t) {
#ifdef DEBUG
    std::cerr << "*************** TCPServer::checkClients ***************" << std::endl;
#endif
  
    //! file descriptor set
    fd_set fdset;

    struct timeval tv;
      
    // Set how long to block.
    tv.tv_sec = wait_t;
    tv.tv_usec = 0;
    
    FD_ZERO(&fdset);
    FD_SET(_master, &fdset);
    
    fd_set errorSet;
    FD_ZERO(&errorSet);
    FD_SET(_master, &errorSet);
    
    select(_master+1, &fdset, NULL, &errorSet, (struct timeval *)&tv);

    // If _master is set then someone is trying to connect
    if(FD_ISSET(_master, &fdset) && !FD_ISSET(_master, &errorSet)) {
      // Open up new connection
      struct sockaddr_in addr;
      int len = sizeof(addr);
#ifdef WIN32
      int client = accept(_master, (struct sockaddr *)&addr, &len);
#else
      int client = accept(_master, (struct sockaddr *)&addr, (socklen_t *)&len);
#endif
      
      if (client == -1) { 
        throw ServerConnectionException("Failed to accept connection!");
      }
  
#ifdef DEBUG
      struct in_addr ip_address;

      // memcpy is used to bypass type casting
      memcpy(&ip_address, &addr.sin_addr.s_addr, 4);
      
      std::cerr << "\n\n---------------------------------------------\n";
      std::cerr << "Connection from: " << inet_ntoa(ip_address) << "  ("
                << ntohs(addr.sin_port) << ")\n";
#endif
      
      
      // spawn thread, bind fd (file descriptor) parameter onto first of server object call
      // to operator()

#ifdef ENABLE_THREADED_SERVER
#ifdef DEBUG
      std::cerr << "Creating Thread" << std::endl;
#endif /* DEBUG */
#ifdef DEBUG
      std::cerr << "Using Threaded code path" << std::endl;
#endif
      // todo: replace with pthreads direct implementation
              
      tcp_thread_hider f(_serverObject, client);
#ifdef MYNAHSA_USE_BOOST
      boost::thread myThread(f);
#else
      Thread< tcp_thread_hider > myThread(f);
#endif
#else
#ifdef DEBUG
      std::cerr << "Using single-thread code path" << std::endl;
#endif
      // enable this and disable above for single thread implementation
      _serverObject(client);
#endif
    }
  }
  
  void TCPServer::bindPort()  {
    int on = 1;
    struct sockaddr_in addr;
    
    _master = socket(AF_INET, SOCK_STREAM, 0);
    memset(&addr, 0, sizeof(addr));
  
#ifdef WIN32
    int sockres = setsockopt(_master, SOL_SOCKET, SO_REUSEADDR, (const char*) &on, sizeof(on));
#else
    int sockres = setsockopt(_master, SOL_SOCKET, SO_REUSEADDR, &on, sizeof(on));
#endif
  
    if ( sockres != 0) { 
      throw ServerException("Setsockopt failed");
    }

    addr.sin_family = AF_INET;
    addr.sin_port = htons(_port);
    addr.sin_addr.s_addr = INADDR_ANY;

    // Open the socket
    if (bind(_master, (struct sockaddr *)&addr, sizeof(addr)) != 0) {
      throw ServerException("Bind failed");
    }
    
    // Set a limit on connection queue.
    if(listen(_master, 5) != 0) {
      throw ServerException("Listen failed");
    }    
  }
};
