/************************************************************************************
 *    This file is part of the MynahSA streaming and archiving toolkit              *
 *    Copyright (C) 2006 Mynah-Software Ltd. All Rights Reserved.                   *
 *                                                                                  *
 *    This program is free software; you can redistribute it and/or modify          *
 *    it under the terms of the GNU General Public License, version 2               *
 *    as published by the Free Software Foundation.                                 *
 *                                                                                  *
 *    This program is distributed in the hope that it will be useful,               *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of                *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                 *
 *    GNU General Public License for more details.                                  *
 *                                                                                  *
 *    You should have received a copy of the GNU General Public License along       *
 *    with this program; if not, write to the Free Software Foundation, Inc.,       *
 *    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.                   *
 *                                                                                  *
 ************************************************************************************/

#include <iostream>
#include <sstream>

#include <mynahsa/mynahsa.hpp>

#include "testcontainer.hpp"

#include <openssl/bio.h>

#include "testserverthread.hpp"

#include "tcptestserverthread.hpp"

#include "testcontainerstreamconstructor.hpp"

#ifndef WIN32
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <unistd.h>
#else
#include <windows.h>
#include <winsock2.h>
#endif


#ifdef WIN32
// work around for lack of win32 sleep
void sleep(int s) {
  Sleep(s*1000);
}

typedef unsigned long in_addr_t;

#endif


using namespace std;
using namespace MynahSA;

int main(int argc, char** argv) { 

  // initialize streaming library
  saInit();

  try { 
    
  {
    // The first step we test text stream archiving.  This test is straight forward
    // and uncomplicated.  Simply place the test object on the stream, and attempt to recover it
    
    cerr << "Testing archive on text stream: ";
    TestContainer tc1;
    // populate test data
    tc1.populateTestData();
    
    ostringstream os;
    OTextStream ots(os);
    OArchive<OTextStream> osst(ots);
    osst << tc1;  
  
  
    // create an input stream
    istringstream is(os.str());
    ITextStream its(is);
    IArchive<ITextStream> iats(its);
    TestContainer tc2;
      
    iats >> tc2;
    if (tc1 == tc2) { 
      cerr << "passed" << endl;
    } else { 
      cerr << "failed" << endl;
      tc1.testCompare(tc2);
      tc2.testPrint();
    }
  }
  
  {
    // Binary archiving is similar to text archiving.
    cerr << "Testing archive on binary stream: ";
    TestContainer tc1;
    // populate test data
    tc1.populateTestData();
    
    ostringstream os;
    OBinaryStream ots(os);
    OArchive<OBinaryStream> osst(ots);
    osst << tc1;  
  
  
    // create an input stream
    istringstream is(os.str());
    IBinaryStream its(is);
    IArchive<IBinaryStream> iats(its);
    TestContainer tc2;
      
    iats >> tc2;
    if (tc1 == tc2) { 
      cerr << "passed" << endl;
    } else { 
      cerr << "failed" << endl;
    }
  }
  
  {
    // Binary archiving is similar to text archiving.
    cerr << "Testing archive on blowfish stream: ";
    TestContainer tc1;
    // populate test data
    tc1.populateTestData();
    
    ostringstream os;

    // Note: OBlowfishStreams need to be flushed.  Closing is the only way to flush
    OBlowfishStream ots(os, "This is the key");
    OArchive<OBlowfishStream> osst(ots);
    osst << tc1;  
    ots.close();

  
    // create an input stream
    istringstream is(os.str());
    IBlowfishStream its(is, "This is the key");
    IArchive<IBlowfishStream> iats(its);
    TestContainer tc2;
      
    iats >> tc2;
    if (tc1 == tc2) { 
      cerr << "passed" << endl;
    } else { 
      cerr << "failed" << endl;
    }
  }
  
  
  
  { 
    cerr << "Beginning SSLRPC Client/Server archive test: ";
    // ensure that SSL subsystem is started.
    sslInit();
    // start a server thread
    bool running = true;  // running acts as a signal
    TestServerThread s(running);
#ifdef MYNAHSA_USE_BOOST
    boost::thread myThread(s);
#else
    Thread<TestServerThread> myThread(s);  // server should be started at this point
#endif
    
    sleep(5);  // allow thread to start
    
    // now create the client
    in_addr_t ina;
#ifndef WIN32
    inet_aton("127.0.0.1", (struct in_addr*) &ina);
#else
    ina = inet_addr("127.0.0.1");
#endif
    TestContainerStreamConstructor myConstructor;
    

    SSLRPCClient myClient(myConstructor, ina, 7501);

    
    SHARED_PTR<TestContainer> tc1p(new TestContainer);
    tc1p->populateTestData();

    SHARED_PTR<IoBase> resp(myClient.rpc(tc1p));
    if (resp->ioName() == TestContainer().ioName()) {
      SHARED_PTR<TestContainer> tc2p(STATIC_PTR_CAST<TestContainer>(resp));
      if ( (*tc1p) == (*tc2p) ) { 
        cerr << "SSL Test passed.";
      } else { 
        cerr << "SSL Test failed."; 
      }
    } else { 
      cerr << "SSL Test Failed to recover object" << endl;
    }
    
    // shutdown the server thread

    running = false; 
    myThread.join();  // join the thread
    
    cerr << endl;  

  }
  
  sleep(1);

  { 
    cerr << "Beginning TCPRPC Client/Server archive test: ";
    // start a server thread
    bool running = true;  // running acts as a signal
        
    TCPTestServerThread s(running);
#ifdef MYNAHSA_USE_BOOST
    boost::thread myThread(s);
#else
    Thread<TCPTestServerThread> myThread(s);  // server should be started at this point
#endif
    
    sleep(1);  // allow thread to start
    
    // now create the client
    in_addr_t ina;

#ifndef WIN32
    inet_aton("127.0.0.1", (struct in_addr*) &ina);
#else
    ina = inet_addr("127.0.0.1");
#endif
    
    TestContainerStreamConstructor myConstructor;
     
    TCPRPCClient myClient(myConstructor, ina, 7501);
    
    
    SHARED_PTR<TestContainer> tc1p(new TestContainer);
    tc1p->populateTestData();

    SHARED_PTR<IoBase> resp(myClient.rpc(tc1p));
    if (resp->ioName() == "TestContainer") {
      SHARED_PTR<TestContainer> tc2p(STATIC_PTR_CAST<TestContainer>(resp));
      if ( (*tc1p) == (*tc2p) ) { 
        cerr << "TCP Test passed.";
      } else { 
        cerr << "TCP Test failed."; 
      }
    } else { 
      cerr << "TCP Test Failed to recover object" << endl;
    }
    
    // shutdown the server thread

    running = false; 
    myThread.join();  // join the thread
    
    cerr << endl;  

  }
  
  } catch(ExceptionBase& eb) { 
    cerr << "Caught a MynahSA::ExecptionBase: " << eb.what() << endl;
    return 1;
  }catch (...) { 
    cerr << "Caught an unhandled exception - this is a failure!" << endl;
    return 1;
  }
  
  return 0;
}
