/************************************************************************************
 *    This file is part of the MynahSA streaming and archiving toolkit              *
 *    Copyright (C) 2006 Mynah-Software Ltd. All Rights Reserved.                   *
 *                                                                                  *
 *    This program is free software; you can redistribute it and/or modify          *
 *    it under the terms of the GNU General Public License, version 2               *
 *    as published by the Free Software Foundation.                                 *
 *                                                                                  *
 *    This program is distributed in the hope that it will be useful,               *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of                *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                 *
 *    GNU General Public License for more details.                                  *
 *                                                                                  *
 *    You should have received a copy of the GNU General Public License along       *
 *    with this program; if not, write to the Free Software Foundation, Inc.,       *
 *    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.                   *
 *                                                                                  *
 ************************************************************************************/

#include <iostream>
#include "testcontainer.hpp"

#include <mynahsa/archive.hpp>

#include <algorithm>
#include <iterator>

using namespace std;
using namespace MynahSA;

TestContainer::TestContainer() : testChar(0),
                                 testUChar(0),
                                 testShort(0),
                                 testUShort(0),
                                 testInt(0),
                                 testUInt(0),
                                 testLongLong(0),
                                 testULongLong(0),
                                 testDouble(0.0),
                                 testFloat(0.0),
                                 _fundamentalInt(new int),
                                 _a(new int),
                                 _aCopy(_a) {
  (*_fundamentalInt) = 0;                                 
  (*_a) = 0;
  // clear testArray
  for (unsigned int i=0; i<sizeof(_testArray); i++) { 
    _testArray[i] = 0;
  }
}
  
TestContainer::~TestContainer() {
// nothing to do
}

void TestContainer::populateTestData() {
  // fill in test data here
  testChar = 't';
  testUChar = 152;
  testShort = -12345;
  testUShort = 65532; 
  testInt = -10243274;
  testUInt = 0x87654321;
  //BMS20060622 - added ll extension, MacOSX wouldn't accept otherwise.
  testLongLong = 0x0123456789abcdef0ll;
  testULongLong = 0xdeadbeefdeadbeefll;
  
  // NOTE: For both the double and floating point values we truncate the 
  //       precision.  This allows the test to pass by writing the PI approximation
  //       onto a text stream without precision loss
  
  testDouble = 3.1415;
  testFloat = 3.1415;
  testString = "Hello World, this is a test string";
  testIntVec.push_back(1);
  testIntVec.push_back(2);
  testIntVec.push_back(3);
  testIntVec.push_back(4);
  
  testStringList.push_back("Hello");
  testStringList.push_back("World");
  testStringList.push_back("My");
  testStringList.push_back("name");
  testStringList.push_back("is");
  testStringList.push_back("Brett.");
  
  // load the test pair
  testPair.first = "Brett Saunders";
  testPair.second = 32;
  
  testMap["Brett"] = 32;
  testMap["Agi"] = 35;
  testMap["John"] = 28;
  testMap["Tupac"] = -1;
  
  (*_fundamentalInt) = 123;  // cannot be zero
  (*_a) = 3; // cannot be zero
  
  _testSet.insert(1);
  _testSet.insert(3);
  _testSet.insert(2);
  _testSet.insert(4);
  
  _testArray[0] = 'B';
  _testArray[1] = 'r';
  _testArray[2] = 'e';
  _testArray[3] = 't';
  _testArray[4] = 't';
  _testArray[5] = 0;   // terminate c string
  
}

bool TestContainer::operator==(const TestContainer& tc) { 
  bool primary = ( testChar       == tc.testChar &&
                   testUChar      == tc.testUChar &&
                   testShort      == tc.testShort &&
                   testUShort     == tc.testUShort &&
                   testInt        == tc.testInt &&
                   testUInt       == tc.testUInt &&
                   testLongLong   == tc.testLongLong &&
                   testULongLong  == tc.testULongLong &&
                   testDouble     == tc.testDouble &&
                   testFloat      == tc.testFloat &&
                   testString     == tc.testString &&
                   testIntVec     == tc.testIntVec &&
                   testStringList == tc.testStringList &&
                   testPair       == tc.testPair &&
                   testMap        == tc.testMap &&
                   (*_fundamentalInt) == (*(tc._fundamentalInt)) &&
                   _a == _aCopy &&   // this test is inserted for pointer equivalence
                   (*_a) == (*(tc._a)) &&
                   _testSet == tc._testSet
                 );
  if (!primary) {
    return false;
  } else { 
    for (unsigned int i=0; i<sizeof(_testArray); i++) { 
      if (_testArray[i] != tc._testArray[i]) { 
        return false;
      }
    }
    return true;
  }                   
}

void TestContainer::serialize(Archive& ar) { 
  ar & testChar;
  ar & testUChar;
  ar & testShort;
  ar & testUShort;
  ar & testInt;
  ar & testUInt;
  ar & testLongLong;
  ar & testULongLong;
  ar & testDouble;
  ar & testFloat;
  ar & testString;
  ar & testIntVec;
  ar & testStringList;
  ar & testPair;
  ar & testMap;
  ar & _fundamentalInt;
  ar & _a;
  ar & _aCopy;
  ar & _testSet;
  // Note: this direct call to operator& is a work-around for the SUNpro 5.8 compiler. GCC correctly interprets it
  //       as the use of operator& on the _testArray type; so on GCC writing ar & _testArray is perfectly legal.
  ar.operator&( _testArray );
}

void TestContainer::testPrint() const { 
  cerr << "testChar (as integer): " << ((int) testChar) << endl;
  cerr << "testUChar (as integer): " << ((unsigned int) testUChar) << endl;
  cerr << "testShort: " << testShort << endl;
  cerr << "testUShort: " << testUShort << endl;
  cerr << "testInt: " << testInt << endl;
  cerr << "testUInt: " << testUInt << endl;
  cerr << "testLongLong: " << testLongLong << endl;
  cerr << "testULongLong: " << testULongLong << endl;
  cerr << "testDouble: " << testDouble << endl;
  cerr << "testFloat: " << testFloat << endl;
  cerr << "testString: " << testString << endl;
  cerr << "testIntVec: ";
  copy(testIntVec.begin(), testIntVec.end(), ostream_iterator<int>(cerr, ","));
  cerr << endl;
  
  cerr << "testStringList: ";
  copy(testStringList.begin(), testStringList.end(), ostream_iterator<string>(cerr, ","));
  cerr << endl;
  
  cerr << "Name: " << testPair.first << " is " << testPair.second << " years old." << endl;
  
  
  cerr << "Test map: " << endl;
  for (map<string,int>::const_iterator mit = testMap.begin();
       mit != testMap.end();
       ++mit) {
    cerr << "  " << (*mit).first << " is:  " << (*mit).second << " years old" << endl;   
  }
  
  cerr << "_fundamentalInt points to an integer of value: " << (*_fundamentalInt) << endl;
  cerr << "_a points to an integer of value: " << (*_a) << endl;
  cerr << "_a == _aCopy is: " << (_a == _aCopy) << endl;
  
  cerr << "Test set: ";
  copy(_testSet.begin(),_testSet.end(), ostream_iterator<int>(cerr, ", "));
  cerr << endl;
  
  cerr << "Test array is: " << _testArray << endl;
  
}

#define MEMBER_TEST( x )   cerr << #x ": "; \
  if ( x == tc.x) { \
    cerr << "ok" << endl; \
  } else { \
    cerr << "failed" << endl; \
  } 

#define MEMBER_PTR_TEST( x )   cerr << "*" #x ": "; \
  if ( (*x) == (*tc.x)) { \
    cerr << "ok" << endl; \
  } else { \
    cerr << "failed" << endl; \
  } 


void TestContainer::testCompare(const TestContainer& tc) const { 
  MEMBER_TEST( testChar );
  MEMBER_TEST( testUChar );
  MEMBER_TEST( testShort );
  MEMBER_TEST( testUShort );
  MEMBER_TEST( testInt );
  MEMBER_TEST( testUInt );
  MEMBER_TEST( testLongLong );
  MEMBER_TEST( testULongLong );
  MEMBER_TEST( testDouble );
  MEMBER_TEST( testFloat );
  MEMBER_TEST( testString );
  MEMBER_TEST( testIntVec );
  MEMBER_TEST( testStringList );
  MEMBER_TEST( testPair );
  MEMBER_TEST( testMap );
 
  MEMBER_PTR_TEST( _fundamentalInt);
  MEMBER_PTR_TEST( _a );
  
  MEMBER_TEST( _testSet );
  for (unsigned int i=0; i<sizeof(_testArray); i++) {
    MEMBER_TEST(_testArray[i]);
  }
}

#undef MEMBER_TEST

std::string TestContainer::ioName() const {
  return "TestContainer";
}
