/************************************************************************************
 *    This file is part of the MynahSA streaming and archiving toolkit              *
 *    Copyright (C) 2006 Mynah-Software Ltd. All Rights Reserved.                   *
 *                                                                                  *
 *    This program is free software; you can redistribute it and/or modify          *
 *    it under the terms of the GNU General Public License, version 2               *
 *    as published by the Free Software Foundation.                                 *
 *                                                                                  *
 *    This program is distributed in the hope that it will be useful,               *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of                *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                 *
 *    GNU General Public License for more details.                                  *
 *                                                                                  *
 *    You should have received a copy of the GNU General Public License along       *
 *    with this program; if not, write to the Free Software Foundation, Inc.,       *
 *    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.                   *
 *                                                                                  *
 ************************************************************************************/

#ifndef __test_container_hpp
#define __test_container_hpp

#include <mynahsa/iobase.hpp>
#include <mynahsa/archive.hpp>

// shared pointer implementation
#include <mynahsa/spimpl.hpp>

#include <string>

#include <vector>
#include <list>
#include <utility>
#include <map>
#include <set>

/** class TestContainer is intended to contain a sample of each of the archive's supported types.
 *  To verify that the archiver is working correctly, we should be able to create and fill the test container with some
 *  random data, store the object onto a stream.  Using the archiver to instate a new object using the data on the stream,
 *  and comparing the two objects will serve to validate that the data was correctly streamed and recovered.
 */
class TestContainer : public MynahSA::IoBase { 
private:
  // basic types first; std containers later
  
  // integer types
  char testChar;
  unsigned char testUChar;
  
  short testShort;
  unsigned short testUShort;
  
  int testInt;
  unsigned int testUInt;
  
  long long testLongLong;
  unsigned long long testULongLong;
  
  // floating point types
  double testDouble;
  float  testFloat;
  
  
  // string type
  std::string testString;
  
  
  // vector of int
  std::vector<int> testIntVec;
  
  // test string list
  std::list< std::string > testStringList;
  
  // test pair
  std::pair<std::string, int> testPair;
  
  // test map
  std::map<std::string, int> testMap;


  // pointer tests!
  SHARED_PTR<int> _fundamentalInt;
  
  // A - B equivalence - B should point to the same location as B and vice versa

  SHARED_PTR<int> _a;

  SHARED_PTR<int> _aCopy;


  std::set<int> _testSet;
  
  char _testArray[6];
  

public:
  //! constructor - initializes integer/floating point types with default values;
  TestContainer();
  
  //! destructor
  ~TestContainer();

  //! fill in data with non-default test values
  void populateTestData();

  //! comparison operator
  bool operator==(const TestContainer& tc);
  
  //! serialization method
  void serialize(MynahSA::Archive& ar);

  //! return the method name
  std::string ioName() const;
  
  //! dump data onto cerr
  void testPrint() const;

  //! allow us to examine which members are identical or not
  void testCompare(const TestContainer& tc) const;

};


#endif
